// FORMATTING
import { addFormatToken } from '../format/format';
import { zeroFill } from '../utils/zero-fill';
import { isNumber, isString, toInt } from '../utils/type-checks';
import { addRegexToken, matchOffset, matchShortOffset } from '../parse/regex';
import { add } from '../moment/add-subtract';
import { addParseToken } from '../parse/token';
import { cloneDate } from '../create/clone';
import { setMonth } from '../utils/date-setters';
function addOffsetFormatToken(token, separator) {
    addFormatToken(token, null, null, function (date, config) {
        let offset = getUTCOffset(date, { _isUTC: config.isUTC, _offset: config.offset });
        let sign = '+';
        if (offset < 0) {
            offset = -offset;
            sign = '-';
        }
        return sign + zeroFill(~~(offset / 60), 2) + separator + zeroFill(~~(offset) % 60, 2);
    });
}
export function initOffset() {
    addOffsetFormatToken('Z', ':');
    addOffsetFormatToken('ZZ', '');
    // PARSING
    addRegexToken('Z', matchShortOffset);
    addRegexToken('ZZ', matchShortOffset);
    addParseToken(['Z', 'ZZ'], function (input, array, config) {
        config._useUTC = true;
        config._tzm = offsetFromString(matchShortOffset, input);
        return config;
    });
}
// HELPERS
// timezone chunker
// '+10:00' > ['10',  '00']
// '-1530'  > ['-15', '30']
const chunkOffset = /([\+\-]|\d\d)/gi;
function offsetFromString(matcher, str) {
    const matches = (str || '').match(matcher);
    if (matches === null) {
        return null;
    }
    const chunk = matches[matches.length - 1];
    const parts = chunk.match(chunkOffset) || ['-', '0', '0'];
    const minutes = parseInt(parts[1], 10) * 60 + toInt(parts[2]);
    const _min = parts[0] === '+' ? minutes : -minutes;
    return minutes === 0 ? 0 : _min;
}
// Return a moment from input, that is local/utc/zone equivalent to model.
export function cloneWithOffset(input, date, config = {}) {
    if (!config._isUTC) {
        return input;
    }
    const res = cloneDate(date);
    // todo: input._d - res._d + ((res._offset || 0) - (input._offset || 0))*60000
    const offsetDiff = (config._offset || 0) * 60000;
    const diff = input.valueOf() - res.valueOf() + offsetDiff;
    // Use low-level api, because this fn is low-level api.
    res.setTime(res.valueOf() + diff);
    // todo: add timezone handling
    // hooks.updateOffset(res, false);
    return res;
}
export function getDateOffset(date) {
    // On Firefox.24 Date#getTimezoneOffset returns a floating point.
    // https://github.com/moment/moment/pull/1871
    return -Math.round(date.getTimezoneOffset() / 15) * 15;
}
// HOOKS
// This function will be called whenever a moment is mutated.
// It is intended to keep the offset in sync with the timezone.
// todo: it's from moment timezones
// hooks.updateOffset = function () {
// };
// MOMENTS
// keepLocalTime = true means only change the timezone, without
// affecting the local hour. So 5:31:26 +0300 --[utcOffset(2, true)]-->
// 5:31:26 +0200 It is possible that 5:31:26 doesn't exist with offset
// +0200, so we adjust the time as needed, to be valid.
//
// Keeping the time actually adds/subtracts (one hour)
// from the actual represented time. That is why we call updateOffset
// a second time. In case it wants us to change the offset again
// _changeInProgress == true case, then we have to adjust, because
// there is no such time in the given timezone.
export function getUTCOffset(date, config = {}) {
    const _offset = config._offset || 0;
    return config._isUTC ? _offset : getDateOffset(date);
}
export function setUTCOffset(date, input, keepLocalTime, keepMinutes, config = {}) {
    const offset = config._offset || 0;
    let localAdjust;
    let _input = input;
    let _date = date;
    if (isString(_input)) {
        _input = offsetFromString(matchShortOffset, _input);
        if (_input === null) {
            return _date;
        }
    }
    else if (isNumber(_input) && Math.abs(_input) < 16 && !keepMinutes) {
        _input = _input * 60;
    }
    if (!config._isUTC && keepLocalTime) {
        localAdjust = getDateOffset(_date);
    }
    config._offset = _input;
    config._isUTC = true;
    if (localAdjust != null) {
        _date = add(_date, localAdjust, 'minutes');
    }
    if (offset !== _input) {
        if (!keepLocalTime || config._changeInProgress) {
            _date = add(_date, _input - offset, 'minutes', config._isUTC);
            // addSubtract(this, createDuration(_input - offset, 'm'), 1, false);
        }
        else if (!config._changeInProgress) {
            config._changeInProgress = true;
            // todo: add timezone handling
            // hooks.updateOffset(this, true);
            config._changeInProgress = null;
        }
    }
    return _date;
}
/*
export function getSetZone(input, keepLocalTime) {
  if (input != null) {
    if (typeof input !== 'string') {
      input = -input;
    }

    this.utcOffset(input, keepLocalTime);

    return this;
  } else {
    return -this.utcOffset();
  }
}
*/
export function setOffsetToUTC(date, keepLocalTime) {
    return setUTCOffset(date, 0, keepLocalTime);
}
export function isDaylightSavingTime(date) {
    return (getUTCOffset(date) > getUTCOffset(setMonth(cloneDate(date), 0))
        || getUTCOffset(date) > getUTCOffset(setMonth(cloneDate(date), 5)));
}
/*export function setOffsetToLocal(date: Date, isUTC?: boolean, keepLocalTime?: boolean) {
  if (this._isUTC) {
    this.utcOffset(0, keepLocalTime);
    this._isUTC = false;

    if (keepLocalTime) {
      this.subtract(getDateOffset(this), 'm');
    }
  }
  return this;
}*/
export function setOffsetToParsedOffset(date, input, config = {}) {
    if (config._tzm != null) {
        return setUTCOffset(date, config._tzm, false, true, config);
    }
    if (isString(input)) {
        const tZone = offsetFromString(matchOffset, input);
        if (tZone != null) {
            return setUTCOffset(date, tZone, false, false, config);
        }
        return setUTCOffset(date, 0, true, false, config);
    }
    return date;
}
export function hasAlignedHourOffset(date, input) {
    const _input = input ? getUTCOffset(input, { _isUTC: false }) : 0;
    return (getUTCOffset(date) - _input) % 60 === 0;
}
// DEPRECATED
/*export function isDaylightSavingTimeShifted() {
  if (!isUndefined(this._isDSTShifted)) {
    return this._isDSTShifted;
  }

  const c = {};

  copyConfig(c, this);
  c = prepareConfig(c);

  if (c._a) {
    const other = c._isUTC ? createUTC(c._a) : createLocal(c._a);
    this._isDSTShifted = this.isValid() &&
      compareArrays(c._a, other.toArray()) > 0;
  } else {
    this._isDSTShifted = false;
  }

  return this._isDSTShifted;
}*/
// in Khronos
/*export function isLocal() {
  return this.isValid() ? !this._isUTC : false;
}

export function isUtcOffset() {
  return this.isValid() ? this._isUTC : false;
}

export function isUtc() {
  return this.isValid() ? this._isUTC && this._offset === 0 : false;
}*/
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoib2Zmc2V0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vLi4vLi4vc3JjL2Nocm9ub3MvdW5pdHMvb2Zmc2V0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFBLGFBQWE7QUFFYixPQUFPLEVBQUUsY0FBYyxFQUFFLE1BQU0sa0JBQWtCLENBQUM7QUFDbEQsT0FBTyxFQUFFLFFBQVEsRUFBRSxNQUFNLG9CQUFvQixDQUFDO0FBRTlDLE9BQU8sRUFBRSxRQUFRLEVBQUUsUUFBUSxFQUFFLEtBQUssRUFBRSxNQUFNLHNCQUFzQixDQUFDO0FBQ2pFLE9BQU8sRUFBRSxhQUFhLEVBQUUsV0FBVyxFQUFFLGdCQUFnQixFQUFFLE1BQU0sZ0JBQWdCLENBQUM7QUFDOUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxNQUFNLHdCQUF3QixDQUFDO0FBQzdDLE9BQU8sRUFBRSxhQUFhLEVBQUUsTUFBTSxnQkFBZ0IsQ0FBQztBQUUvQyxPQUFPLEVBQUUsU0FBUyxFQUFFLE1BQU0saUJBQWlCLENBQUM7QUFDNUMsT0FBTyxFQUFFLFFBQVEsRUFBRSxNQUFNLHVCQUF1QixDQUFDO0FBRWpELFNBQVMsb0JBQW9CLENBQUMsS0FBYSxFQUFFLFNBQWlCO0lBQzVELGNBQWMsQ0FBQyxLQUFLLEVBQUUsSUFBSSxFQUFFLElBQUksRUFBRSxVQUFVLElBQVUsRUFBRSxNQUFNO1FBQzVELElBQUksTUFBTSxHQUFHLFlBQVksQ0FBQyxJQUFJLEVBQUUsRUFBQyxNQUFNLEVBQUUsTUFBTSxDQUFDLEtBQUssRUFBRSxPQUFPLEVBQUUsTUFBTSxDQUFDLE1BQU0sRUFBQyxDQUFDLENBQUM7UUFDaEYsSUFBSSxJQUFJLEdBQUcsR0FBRyxDQUFDO1FBQ2YsSUFBSSxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQ2QsTUFBTSxHQUFHLENBQUMsTUFBTSxDQUFDO1lBQ2pCLElBQUksR0FBRyxHQUFHLENBQUM7U0FDWjtRQUVELE9BQU8sSUFBSSxHQUFHLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxNQUFNLEdBQUcsRUFBRSxDQUFDLEVBQUUsQ0FBQyxDQUFDLEdBQUcsU0FBUyxHQUFHLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUM7SUFDeEYsQ0FBQyxDQUFDLENBQUM7QUFDTCxDQUFDO0FBRUQsTUFBTSxVQUFVLFVBQVU7SUFDeEIsb0JBQW9CLENBQUMsR0FBRyxFQUFFLEdBQUcsQ0FBQyxDQUFDO0lBQy9CLG9CQUFvQixDQUFDLElBQUksRUFBRSxFQUFFLENBQUMsQ0FBQztJQUVqQyxVQUFVO0lBRVIsYUFBYSxDQUFDLEdBQUcsRUFBRSxnQkFBZ0IsQ0FBQyxDQUFDO0lBQ3JDLGFBQWEsQ0FBQyxJQUFJLEVBQUUsZ0JBQWdCLENBQUMsQ0FBQztJQUN0QyxhQUFhLENBQUMsQ0FBQyxHQUFHLEVBQUUsSUFBSSxDQUFDLEVBQUUsVUFBUyxLQUFhLEVBQUUsS0FBZ0IsRUFBRSxNQUF5QjtRQUM1RixNQUFNLENBQUMsT0FBTyxHQUFHLElBQUksQ0FBQztRQUN0QixNQUFNLENBQUMsSUFBSSxHQUFHLGdCQUFnQixDQUFDLGdCQUFnQixFQUFFLEtBQUssQ0FBQyxDQUFDO1FBRXhELE9BQU8sTUFBTSxDQUFDO0lBQ2hCLENBQUMsQ0FBQyxDQUFDO0FBQ0wsQ0FBQztBQUVELFVBQVU7QUFFVixtQkFBbUI7QUFDbkIsMkJBQTJCO0FBQzNCLDJCQUEyQjtBQUMzQixNQUFNLFdBQVcsR0FBRyxpQkFBaUIsQ0FBQztBQUV0QyxTQUFTLGdCQUFnQixDQUFDLE9BQWUsRUFBRSxHQUFXO0lBQ3BELE1BQU0sT0FBTyxHQUFHLENBQUMsR0FBRyxJQUFJLEVBQUUsQ0FBQyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQztJQUUzQyxJQUFJLE9BQU8sS0FBSyxJQUFJLEVBQUU7UUFDcEIsT0FBTyxJQUFJLENBQUM7S0FDYjtJQUVELE1BQU0sS0FBSyxHQUFHLE9BQU8sQ0FBQyxPQUFPLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDO0lBQzFDLE1BQU0sS0FBSyxHQUFHLEtBQUssQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUUsR0FBRyxFQUFFLEdBQUcsQ0FBQyxDQUFDO0lBQzFELE1BQU0sT0FBTyxHQUFHLFFBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLEdBQUcsRUFBRSxHQUFHLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUM5RCxNQUFNLElBQUksR0FBRyxLQUFLLENBQUMsQ0FBQyxDQUFDLEtBQUssR0FBRyxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDO0lBRW5ELE9BQU8sT0FBTyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUM7QUFDbEMsQ0FBQztBQUVELDBFQUEwRTtBQUMxRSxNQUFNLFVBQVUsZUFBZSxDQUFDLEtBQVcsRUFBRSxJQUFVLEVBQ3ZCLFNBQTRCLEVBQUU7SUFDNUQsSUFBSSxDQUFDLE1BQU0sQ0FBQyxNQUFNLEVBQUU7UUFDbEIsT0FBTyxLQUFLLENBQUM7S0FDZDtJQUVELE1BQU0sR0FBRyxHQUFHLFNBQVMsQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUM1Qiw4RUFBOEU7SUFDOUUsTUFBTSxVQUFVLEdBQUcsQ0FBQyxNQUFNLENBQUMsT0FBTyxJQUFJLENBQUMsQ0FBQyxHQUFHLEtBQUssQ0FBQztJQUNqRCxNQUFNLElBQUksR0FBRyxLQUFLLENBQUMsT0FBTyxFQUFFLEdBQUcsR0FBRyxDQUFDLE9BQU8sRUFBRSxHQUFHLFVBQVUsQ0FBQztJQUMxRCx1REFBdUQ7SUFDdkQsR0FBRyxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsT0FBTyxFQUFFLEdBQUcsSUFBSSxDQUFDLENBQUM7SUFDbEMsOEJBQThCO0lBQzlCLGtDQUFrQztJQUVsQyxPQUFPLEdBQUcsQ0FBQztBQUNiLENBQUM7QUFFRCxNQUFNLFVBQVUsYUFBYSxDQUFDLElBQVU7SUFDdEMsaUVBQWlFO0lBQ2pFLDZDQUE2QztJQUM3QyxPQUFPLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsaUJBQWlCLEVBQUUsR0FBRyxFQUFFLENBQUMsR0FBRyxFQUFFLENBQUM7QUFDekQsQ0FBQztBQUVELFFBQVE7QUFFUiw2REFBNkQ7QUFDN0QsK0RBQStEO0FBQy9ELG1DQUFtQztBQUNuQyxxQ0FBcUM7QUFDckMsS0FBSztBQUVMLFVBQVU7QUFFViwrREFBK0Q7QUFDL0QsdUVBQXVFO0FBQ3ZFLHNFQUFzRTtBQUN0RSx1REFBdUQ7QUFDdkQsRUFBRTtBQUNGLHNEQUFzRDtBQUN0RCxxRUFBcUU7QUFDckUsZ0VBQWdFO0FBQ2hFLGtFQUFrRTtBQUNsRSwrQ0FBK0M7QUFDL0MsTUFBTSxVQUFVLFlBQVksQ0FBQyxJQUFVLEVBQUUsU0FBNEIsRUFBRTtJQUNyRSxNQUFNLE9BQU8sR0FBRyxNQUFNLENBQUMsT0FBTyxJQUFJLENBQUMsQ0FBQztJQUVwQyxPQUFPLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxDQUFDO0FBQ3ZELENBQUM7QUFFRCxNQUFNLFVBQVUsWUFBWSxDQUFDLElBQVUsRUFBRSxLQUFzQixFQUFFLGFBQXVCLEVBQUUsV0FBcUIsRUFBRSxTQUE0QixFQUFFO0lBQzdJLE1BQU0sTUFBTSxHQUFHLE1BQU0sQ0FBQyxPQUFPLElBQUksQ0FBQyxDQUFDO0lBQ25DLElBQUksV0FBVyxDQUFDO0lBQ2hCLElBQUksTUFBTSxHQUFHLEtBQUssQ0FBQztJQUNuQixJQUFJLEtBQUssR0FBRyxJQUFJLENBQUM7SUFFakIsSUFBSSxRQUFRLENBQUMsTUFBTSxDQUFDLEVBQUU7UUFDcEIsTUFBTSxHQUFHLGdCQUFnQixDQUFDLGdCQUFnQixFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBQ3BELElBQUksTUFBTSxLQUFLLElBQUksRUFBRTtZQUNuQixPQUFPLEtBQUssQ0FBQztTQUNkO0tBQ0Y7U0FBTSxJQUFJLFFBQVEsQ0FBQyxNQUFNLENBQUMsSUFBSSxJQUFJLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsSUFBSSxDQUFDLFdBQVcsRUFBRTtRQUNwRSxNQUFNLEdBQUcsTUFBTSxHQUFHLEVBQUUsQ0FBQztLQUN0QjtJQUVELElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTSxJQUFJLGFBQWEsRUFBRTtRQUNuQyxXQUFXLEdBQUcsYUFBYSxDQUFDLEtBQUssQ0FBQyxDQUFDO0tBQ3BDO0lBQ0QsTUFBTSxDQUFDLE9BQU8sR0FBRyxNQUFNLENBQUM7SUFDeEIsTUFBTSxDQUFDLE1BQU0sR0FBRyxJQUFJLENBQUM7SUFDckIsSUFBSSxXQUFXLElBQUksSUFBSSxFQUFFO1FBQ3ZCLEtBQUssR0FBRyxHQUFHLENBQUMsS0FBSyxFQUFFLFdBQVcsRUFBRSxTQUFTLENBQUMsQ0FBQztLQUM1QztJQUNELElBQUksTUFBTSxLQUFLLE1BQU0sRUFBRTtRQUNyQixJQUFJLENBQUMsYUFBYSxJQUFJLE1BQU0sQ0FBQyxpQkFBaUIsRUFBRTtZQUM5QyxLQUFLLEdBQUcsR0FBRyxDQUFDLEtBQUssRUFBRSxNQUFNLEdBQUcsTUFBTSxFQUFFLFNBQVMsRUFBRSxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUM7WUFDOUQscUVBQXFFO1NBQ3RFO2FBQU0sSUFBSSxDQUFDLE1BQU0sQ0FBQyxpQkFBaUIsRUFBRTtZQUNwQyxNQUFNLENBQUMsaUJBQWlCLEdBQUcsSUFBSSxDQUFDO1lBQ2hDLDhCQUE4QjtZQUM5QixrQ0FBa0M7WUFDbEMsTUFBTSxDQUFDLGlCQUFpQixHQUFHLElBQUksQ0FBQztTQUNqQztLQUNGO0lBRUQsT0FBTyxLQUFLLENBQUM7QUFDZixDQUFDO0FBRUQ7Ozs7Ozs7Ozs7Ozs7O0VBY0U7QUFFRixNQUFNLFVBQVUsY0FBYyxDQUFDLElBQVUsRUFBRSxhQUF1QjtJQUNoRSxPQUFPLFlBQVksQ0FBQyxJQUFJLEVBQUUsQ0FBQyxFQUFFLGFBQWEsQ0FBQyxDQUFDO0FBQzlDLENBQUM7QUFFRCxNQUFNLFVBQVUsb0JBQW9CLENBQUMsSUFBVTtJQUU3QyxPQUFPLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxHQUFHLFlBQVksQ0FBQyxRQUFRLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO1dBQ2xFLFlBQVksQ0FBQyxJQUFJLENBQUMsR0FBRyxZQUFZLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7QUFDeEUsQ0FBQztBQUVEOzs7Ozs7Ozs7O0dBVUc7QUFFSCxNQUFNLFVBQVUsdUJBQXVCLENBQUMsSUFBVSxFQUFFLEtBQWEsRUFBRSxTQUE0QixFQUFFO0lBQy9GLElBQUksTUFBTSxDQUFDLElBQUksSUFBSSxJQUFJLEVBQUU7UUFDdkIsT0FBTyxZQUFZLENBQUMsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLEVBQUUsS0FBSyxFQUFFLElBQUksRUFBRSxNQUFNLENBQUMsQ0FBQztLQUM3RDtJQUVELElBQUksUUFBUSxDQUFDLEtBQUssQ0FBQyxFQUFFO1FBQ25CLE1BQU0sS0FBSyxHQUFHLGdCQUFnQixDQUFDLFdBQVcsRUFBRSxLQUFLLENBQUMsQ0FBQztRQUNuRCxJQUFJLEtBQUssSUFBSSxJQUFJLEVBQUU7WUFDakIsT0FBTyxZQUFZLENBQUMsSUFBSSxFQUFFLEtBQUssRUFBRSxLQUFLLEVBQUUsS0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1NBQ3hEO1FBRUQsT0FBTyxZQUFZLENBQUMsSUFBSSxFQUFFLENBQUMsRUFBRSxJQUFJLEVBQUUsS0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFDO0tBQ25EO0lBRUQsT0FBTyxJQUFJLENBQUM7QUFDZCxDQUFDO0FBRUQsTUFBTSxVQUFVLG9CQUFvQixDQUFDLElBQVUsRUFBRSxLQUFZO0lBQzNELE1BQU0sTUFBTSxHQUFHLEtBQUssQ0FBQyxDQUFDLENBQUMsWUFBWSxDQUFDLEtBQUssRUFBRSxFQUFFLE1BQU0sRUFBRSxLQUFLLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFFbEUsT0FBTyxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsR0FBRyxNQUFNLENBQUMsR0FBRyxFQUFFLEtBQUssQ0FBQyxDQUFDO0FBQ2xELENBQUM7QUFHRCxhQUFhO0FBQ2I7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7R0FtQkc7QUFFSCxhQUFhO0FBQ2I7Ozs7Ozs7Ozs7R0FVRyIsInNvdXJjZXNDb250ZW50IjpbIi8vIEZPUk1BVFRJTkdcblxuaW1wb3J0IHsgYWRkRm9ybWF0VG9rZW4gfSBmcm9tICcuLi9mb3JtYXQvZm9ybWF0JztcbmltcG9ydCB7IHplcm9GaWxsIH0gZnJvbSAnLi4vdXRpbHMvemVyby1maWxsJztcbmltcG9ydCB7IERhdGVQYXJzaW5nQ29uZmlnIH0gZnJvbSAnLi4vY3JlYXRlL3BhcnNpbmcudHlwZXMnO1xuaW1wb3J0IHsgaXNOdW1iZXIsIGlzU3RyaW5nLCB0b0ludCB9IGZyb20gJy4uL3V0aWxzL3R5cGUtY2hlY2tzJztcbmltcG9ydCB7IGFkZFJlZ2V4VG9rZW4sIG1hdGNoT2Zmc2V0LCBtYXRjaFNob3J0T2Zmc2V0IH0gZnJvbSAnLi4vcGFyc2UvcmVnZXgnO1xuaW1wb3J0IHsgYWRkIH0gZnJvbSAnLi4vbW9tZW50L2FkZC1zdWJ0cmFjdCc7XG5pbXBvcnQgeyBhZGRQYXJzZVRva2VuIH0gZnJvbSAnLi4vcGFyc2UvdG9rZW4nO1xuaW1wb3J0IHsgRGF0ZUFycmF5IH0gZnJvbSAnLi4vdHlwZXMnO1xuaW1wb3J0IHsgY2xvbmVEYXRlIH0gZnJvbSAnLi4vY3JlYXRlL2Nsb25lJztcbmltcG9ydCB7IHNldE1vbnRoIH0gZnJvbSAnLi4vdXRpbHMvZGF0ZS1zZXR0ZXJzJztcblxuZnVuY3Rpb24gYWRkT2Zmc2V0Rm9ybWF0VG9rZW4odG9rZW46IHN0cmluZywgc2VwYXJhdG9yOiBzdHJpbmcpOiB2b2lkIHtcbiAgYWRkRm9ybWF0VG9rZW4odG9rZW4sIG51bGwsIG51bGwsIGZ1bmN0aW9uIChkYXRlOiBEYXRlLCBjb25maWcpOiBzdHJpbmcge1xuICAgIGxldCBvZmZzZXQgPSBnZXRVVENPZmZzZXQoZGF0ZSwge19pc1VUQzogY29uZmlnLmlzVVRDLCBfb2Zmc2V0OiBjb25maWcub2Zmc2V0fSk7XG4gICAgbGV0IHNpZ24gPSAnKyc7XG4gICAgaWYgKG9mZnNldCA8IDApIHtcbiAgICAgIG9mZnNldCA9IC1vZmZzZXQ7XG4gICAgICBzaWduID0gJy0nO1xuICAgIH1cblxuICAgIHJldHVybiBzaWduICsgemVyb0ZpbGwofn4ob2Zmc2V0IC8gNjApLCAyKSArIHNlcGFyYXRvciArIHplcm9GaWxsKH5+KG9mZnNldCkgJSA2MCwgMik7XG4gIH0pO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gaW5pdE9mZnNldCgpIHtcbiAgYWRkT2Zmc2V0Rm9ybWF0VG9rZW4oJ1onLCAnOicpO1xuICBhZGRPZmZzZXRGb3JtYXRUb2tlbignWlonLCAnJyk7XG5cbi8vIFBBUlNJTkdcblxuICBhZGRSZWdleFRva2VuKCdaJywgbWF0Y2hTaG9ydE9mZnNldCk7XG4gIGFkZFJlZ2V4VG9rZW4oJ1paJywgbWF0Y2hTaG9ydE9mZnNldCk7XG4gIGFkZFBhcnNlVG9rZW4oWydaJywgJ1paJ10sIGZ1bmN0aW9uKGlucHV0OiBzdHJpbmcsIGFycmF5OiBEYXRlQXJyYXksIGNvbmZpZzogRGF0ZVBhcnNpbmdDb25maWcpOiBEYXRlUGFyc2luZ0NvbmZpZyB7XG4gICAgY29uZmlnLl91c2VVVEMgPSB0cnVlO1xuICAgIGNvbmZpZy5fdHptID0gb2Zmc2V0RnJvbVN0cmluZyhtYXRjaFNob3J0T2Zmc2V0LCBpbnB1dCk7XG5cbiAgICByZXR1cm4gY29uZmlnO1xuICB9KTtcbn1cblxuLy8gSEVMUEVSU1xuXG4vLyB0aW1lem9uZSBjaHVua2VyXG4vLyAnKzEwOjAwJyA+IFsnMTAnLCAgJzAwJ11cbi8vICctMTUzMCcgID4gWyctMTUnLCAnMzAnXVxuY29uc3QgY2h1bmtPZmZzZXQgPSAvKFtcXCtcXC1dfFxcZFxcZCkvZ2k7XG5cbmZ1bmN0aW9uIG9mZnNldEZyb21TdHJpbmcobWF0Y2hlcjogUmVnRXhwLCBzdHI6IHN0cmluZyk6IG51bWJlciB7XG4gIGNvbnN0IG1hdGNoZXMgPSAoc3RyIHx8ICcnKS5tYXRjaChtYXRjaGVyKTtcblxuICBpZiAobWF0Y2hlcyA9PT0gbnVsbCkge1xuICAgIHJldHVybiBudWxsO1xuICB9XG5cbiAgY29uc3QgY2h1bmsgPSBtYXRjaGVzW21hdGNoZXMubGVuZ3RoIC0gMV07XG4gIGNvbnN0IHBhcnRzID0gY2h1bmsubWF0Y2goY2h1bmtPZmZzZXQpIHx8IFsnLScsICcwJywgJzAnXTtcbiAgY29uc3QgbWludXRlcyA9IHBhcnNlSW50KHBhcnRzWzFdLCAxMCkgKiA2MCArIHRvSW50KHBhcnRzWzJdKTtcbiAgY29uc3QgX21pbiA9IHBhcnRzWzBdID09PSAnKycgPyBtaW51dGVzIDogLW1pbnV0ZXM7XG5cbiAgcmV0dXJuIG1pbnV0ZXMgPT09IDAgPyAwIDogX21pbjtcbn1cblxuLy8gUmV0dXJuIGEgbW9tZW50IGZyb20gaW5wdXQsIHRoYXQgaXMgbG9jYWwvdXRjL3pvbmUgZXF1aXZhbGVudCB0byBtb2RlbC5cbmV4cG9ydCBmdW5jdGlvbiBjbG9uZVdpdGhPZmZzZXQoaW5wdXQ6IERhdGUsIGRhdGU6IERhdGUsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIGNvbmZpZzogRGF0ZVBhcnNpbmdDb25maWcgPSB7fSk6IERhdGUge1xuICBpZiAoIWNvbmZpZy5faXNVVEMpIHtcbiAgICByZXR1cm4gaW5wdXQ7XG4gIH1cblxuICBjb25zdCByZXMgPSBjbG9uZURhdGUoZGF0ZSk7XG4gIC8vIHRvZG86IGlucHV0Ll9kIC0gcmVzLl9kICsgKChyZXMuX29mZnNldCB8fCAwKSAtIChpbnB1dC5fb2Zmc2V0IHx8IDApKSo2MDAwMFxuICBjb25zdCBvZmZzZXREaWZmID0gKGNvbmZpZy5fb2Zmc2V0IHx8IDApICogNjAwMDA7XG4gIGNvbnN0IGRpZmYgPSBpbnB1dC52YWx1ZU9mKCkgLSByZXMudmFsdWVPZigpICsgb2Zmc2V0RGlmZjtcbiAgLy8gVXNlIGxvdy1sZXZlbCBhcGksIGJlY2F1c2UgdGhpcyBmbiBpcyBsb3ctbGV2ZWwgYXBpLlxuICByZXMuc2V0VGltZShyZXMudmFsdWVPZigpICsgZGlmZik7XG4gIC8vIHRvZG86IGFkZCB0aW1lem9uZSBoYW5kbGluZ1xuICAvLyBob29rcy51cGRhdGVPZmZzZXQocmVzLCBmYWxzZSk7XG5cbiAgcmV0dXJuIHJlcztcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGdldERhdGVPZmZzZXQoZGF0ZTogRGF0ZSk6IG51bWJlciB7XG4gIC8vIE9uIEZpcmVmb3guMjQgRGF0ZSNnZXRUaW1lem9uZU9mZnNldCByZXR1cm5zIGEgZmxvYXRpbmcgcG9pbnQuXG4gIC8vIGh0dHBzOi8vZ2l0aHViLmNvbS9tb21lbnQvbW9tZW50L3B1bGwvMTg3MVxuICByZXR1cm4gLU1hdGgucm91bmQoZGF0ZS5nZXRUaW1lem9uZU9mZnNldCgpIC8gMTUpICogMTU7XG59XG5cbi8vIEhPT0tTXG5cbi8vIFRoaXMgZnVuY3Rpb24gd2lsbCBiZSBjYWxsZWQgd2hlbmV2ZXIgYSBtb21lbnQgaXMgbXV0YXRlZC5cbi8vIEl0IGlzIGludGVuZGVkIHRvIGtlZXAgdGhlIG9mZnNldCBpbiBzeW5jIHdpdGggdGhlIHRpbWV6b25lLlxuLy8gdG9kbzogaXQncyBmcm9tIG1vbWVudCB0aW1lem9uZXNcbi8vIGhvb2tzLnVwZGF0ZU9mZnNldCA9IGZ1bmN0aW9uICgpIHtcbi8vIH07XG5cbi8vIE1PTUVOVFNcblxuLy8ga2VlcExvY2FsVGltZSA9IHRydWUgbWVhbnMgb25seSBjaGFuZ2UgdGhlIHRpbWV6b25lLCB3aXRob3V0XG4vLyBhZmZlY3RpbmcgdGhlIGxvY2FsIGhvdXIuIFNvIDU6MzE6MjYgKzAzMDAgLS1bdXRjT2Zmc2V0KDIsIHRydWUpXS0tPlxuLy8gNTozMToyNiArMDIwMCBJdCBpcyBwb3NzaWJsZSB0aGF0IDU6MzE6MjYgZG9lc24ndCBleGlzdCB3aXRoIG9mZnNldFxuLy8gKzAyMDAsIHNvIHdlIGFkanVzdCB0aGUgdGltZSBhcyBuZWVkZWQsIHRvIGJlIHZhbGlkLlxuLy9cbi8vIEtlZXBpbmcgdGhlIHRpbWUgYWN0dWFsbHkgYWRkcy9zdWJ0cmFjdHMgKG9uZSBob3VyKVxuLy8gZnJvbSB0aGUgYWN0dWFsIHJlcHJlc2VudGVkIHRpbWUuIFRoYXQgaXMgd2h5IHdlIGNhbGwgdXBkYXRlT2Zmc2V0XG4vLyBhIHNlY29uZCB0aW1lLiBJbiBjYXNlIGl0IHdhbnRzIHVzIHRvIGNoYW5nZSB0aGUgb2Zmc2V0IGFnYWluXG4vLyBfY2hhbmdlSW5Qcm9ncmVzcyA9PSB0cnVlIGNhc2UsIHRoZW4gd2UgaGF2ZSB0byBhZGp1c3QsIGJlY2F1c2Vcbi8vIHRoZXJlIGlzIG5vIHN1Y2ggdGltZSBpbiB0aGUgZ2l2ZW4gdGltZXpvbmUuXG5leHBvcnQgZnVuY3Rpb24gZ2V0VVRDT2Zmc2V0KGRhdGU6IERhdGUsIGNvbmZpZzogRGF0ZVBhcnNpbmdDb25maWcgPSB7fSk6IG51bWJlciB7XG4gIGNvbnN0IF9vZmZzZXQgPSBjb25maWcuX29mZnNldCB8fCAwO1xuXG4gIHJldHVybiBjb25maWcuX2lzVVRDID8gX29mZnNldCA6IGdldERhdGVPZmZzZXQoZGF0ZSk7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBzZXRVVENPZmZzZXQoZGF0ZTogRGF0ZSwgaW5wdXQ6IG51bWJlciB8IHN0cmluZywga2VlcExvY2FsVGltZT86IGJvb2xlYW4sIGtlZXBNaW51dGVzPzogYm9vbGVhbiwgY29uZmlnOiBEYXRlUGFyc2luZ0NvbmZpZyA9IHt9KTogRGF0ZSB7XG4gIGNvbnN0IG9mZnNldCA9IGNvbmZpZy5fb2Zmc2V0IHx8IDA7XG4gIGxldCBsb2NhbEFkanVzdDtcbiAgbGV0IF9pbnB1dCA9IGlucHV0O1xuICBsZXQgX2RhdGUgPSBkYXRlO1xuXG4gIGlmIChpc1N0cmluZyhfaW5wdXQpKSB7XG4gICAgX2lucHV0ID0gb2Zmc2V0RnJvbVN0cmluZyhtYXRjaFNob3J0T2Zmc2V0LCBfaW5wdXQpO1xuICAgIGlmIChfaW5wdXQgPT09IG51bGwpIHtcbiAgICAgIHJldHVybiBfZGF0ZTtcbiAgICB9XG4gIH0gZWxzZSBpZiAoaXNOdW1iZXIoX2lucHV0KSAmJiBNYXRoLmFicyhfaW5wdXQpIDwgMTYgJiYgIWtlZXBNaW51dGVzKSB7XG4gICAgX2lucHV0ID0gX2lucHV0ICogNjA7XG4gIH1cblxuICBpZiAoIWNvbmZpZy5faXNVVEMgJiYga2VlcExvY2FsVGltZSkge1xuICAgIGxvY2FsQWRqdXN0ID0gZ2V0RGF0ZU9mZnNldChfZGF0ZSk7XG4gIH1cbiAgY29uZmlnLl9vZmZzZXQgPSBfaW5wdXQ7XG4gIGNvbmZpZy5faXNVVEMgPSB0cnVlO1xuICBpZiAobG9jYWxBZGp1c3QgIT0gbnVsbCkge1xuICAgIF9kYXRlID0gYWRkKF9kYXRlLCBsb2NhbEFkanVzdCwgJ21pbnV0ZXMnKTtcbiAgfVxuICBpZiAob2Zmc2V0ICE9PSBfaW5wdXQpIHtcbiAgICBpZiAoIWtlZXBMb2NhbFRpbWUgfHwgY29uZmlnLl9jaGFuZ2VJblByb2dyZXNzKSB7XG4gICAgICBfZGF0ZSA9IGFkZChfZGF0ZSwgX2lucHV0IC0gb2Zmc2V0LCAnbWludXRlcycsIGNvbmZpZy5faXNVVEMpO1xuICAgICAgLy8gYWRkU3VidHJhY3QodGhpcywgY3JlYXRlRHVyYXRpb24oX2lucHV0IC0gb2Zmc2V0LCAnbScpLCAxLCBmYWxzZSk7XG4gICAgfSBlbHNlIGlmICghY29uZmlnLl9jaGFuZ2VJblByb2dyZXNzKSB7XG4gICAgICBjb25maWcuX2NoYW5nZUluUHJvZ3Jlc3MgPSB0cnVlO1xuICAgICAgLy8gdG9kbzogYWRkIHRpbWV6b25lIGhhbmRsaW5nXG4gICAgICAvLyBob29rcy51cGRhdGVPZmZzZXQodGhpcywgdHJ1ZSk7XG4gICAgICBjb25maWcuX2NoYW5nZUluUHJvZ3Jlc3MgPSBudWxsO1xuICAgIH1cbiAgfVxuXG4gIHJldHVybiBfZGF0ZTtcbn1cblxuLypcbmV4cG9ydCBmdW5jdGlvbiBnZXRTZXRab25lKGlucHV0LCBrZWVwTG9jYWxUaW1lKSB7XG4gIGlmIChpbnB1dCAhPSBudWxsKSB7XG4gICAgaWYgKHR5cGVvZiBpbnB1dCAhPT0gJ3N0cmluZycpIHtcbiAgICAgIGlucHV0ID0gLWlucHV0O1xuICAgIH1cblxuICAgIHRoaXMudXRjT2Zmc2V0KGlucHV0LCBrZWVwTG9jYWxUaW1lKTtcblxuICAgIHJldHVybiB0aGlzO1xuICB9IGVsc2Uge1xuICAgIHJldHVybiAtdGhpcy51dGNPZmZzZXQoKTtcbiAgfVxufVxuKi9cblxuZXhwb3J0IGZ1bmN0aW9uIHNldE9mZnNldFRvVVRDKGRhdGU6IERhdGUsIGtlZXBMb2NhbFRpbWU/OiBib29sZWFuKTogRGF0ZSB7XG4gIHJldHVybiBzZXRVVENPZmZzZXQoZGF0ZSwgMCwga2VlcExvY2FsVGltZSk7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBpc0RheWxpZ2h0U2F2aW5nVGltZShkYXRlOiBEYXRlKTogYm9vbGVhbiB7XG5cbiAgcmV0dXJuIChnZXRVVENPZmZzZXQoZGF0ZSkgPiBnZXRVVENPZmZzZXQoc2V0TW9udGgoY2xvbmVEYXRlKGRhdGUpLCAwKSlcbiAgICB8fCBnZXRVVENPZmZzZXQoZGF0ZSkgPiBnZXRVVENPZmZzZXQoc2V0TW9udGgoY2xvbmVEYXRlKGRhdGUpLCA1KSkpO1xufVxuXG4vKmV4cG9ydCBmdW5jdGlvbiBzZXRPZmZzZXRUb0xvY2FsKGRhdGU6IERhdGUsIGlzVVRDPzogYm9vbGVhbiwga2VlcExvY2FsVGltZT86IGJvb2xlYW4pIHtcbiAgaWYgKHRoaXMuX2lzVVRDKSB7XG4gICAgdGhpcy51dGNPZmZzZXQoMCwga2VlcExvY2FsVGltZSk7XG4gICAgdGhpcy5faXNVVEMgPSBmYWxzZTtcblxuICAgIGlmIChrZWVwTG9jYWxUaW1lKSB7XG4gICAgICB0aGlzLnN1YnRyYWN0KGdldERhdGVPZmZzZXQodGhpcyksICdtJyk7XG4gICAgfVxuICB9XG4gIHJldHVybiB0aGlzO1xufSovXG5cbmV4cG9ydCBmdW5jdGlvbiBzZXRPZmZzZXRUb1BhcnNlZE9mZnNldChkYXRlOiBEYXRlLCBpbnB1dDogc3RyaW5nLCBjb25maWc6IERhdGVQYXJzaW5nQ29uZmlnID0ge30pOiBEYXRlIHtcbiAgaWYgKGNvbmZpZy5fdHptICE9IG51bGwpIHtcbiAgICByZXR1cm4gc2V0VVRDT2Zmc2V0KGRhdGUsIGNvbmZpZy5fdHptLCBmYWxzZSwgdHJ1ZSwgY29uZmlnKTtcbiAgfVxuXG4gIGlmIChpc1N0cmluZyhpbnB1dCkpIHtcbiAgICBjb25zdCB0Wm9uZSA9IG9mZnNldEZyb21TdHJpbmcobWF0Y2hPZmZzZXQsIGlucHV0KTtcbiAgICBpZiAodFpvbmUgIT0gbnVsbCkge1xuICAgICAgcmV0dXJuIHNldFVUQ09mZnNldChkYXRlLCB0Wm9uZSwgZmFsc2UsIGZhbHNlLCBjb25maWcpO1xuICAgIH1cblxuICAgIHJldHVybiBzZXRVVENPZmZzZXQoZGF0ZSwgMCwgdHJ1ZSwgZmFsc2UsIGNvbmZpZyk7XG4gIH1cblxuICByZXR1cm4gZGF0ZTtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGhhc0FsaWduZWRIb3VyT2Zmc2V0KGRhdGU6IERhdGUsIGlucHV0PzogRGF0ZSkge1xuICBjb25zdCBfaW5wdXQgPSBpbnB1dCA/IGdldFVUQ09mZnNldChpbnB1dCwgeyBfaXNVVEM6IGZhbHNlIH0pIDogMDtcblxuICByZXR1cm4gKGdldFVUQ09mZnNldChkYXRlKSAtIF9pbnB1dCkgJSA2MCA9PT0gMDtcbn1cblxuXG4vLyBERVBSRUNBVEVEXG4vKmV4cG9ydCBmdW5jdGlvbiBpc0RheWxpZ2h0U2F2aW5nVGltZVNoaWZ0ZWQoKSB7XG4gIGlmICghaXNVbmRlZmluZWQodGhpcy5faXNEU1RTaGlmdGVkKSkge1xuICAgIHJldHVybiB0aGlzLl9pc0RTVFNoaWZ0ZWQ7XG4gIH1cblxuICBjb25zdCBjID0ge307XG5cbiAgY29weUNvbmZpZyhjLCB0aGlzKTtcbiAgYyA9IHByZXBhcmVDb25maWcoYyk7XG5cbiAgaWYgKGMuX2EpIHtcbiAgICBjb25zdCBvdGhlciA9IGMuX2lzVVRDID8gY3JlYXRlVVRDKGMuX2EpIDogY3JlYXRlTG9jYWwoYy5fYSk7XG4gICAgdGhpcy5faXNEU1RTaGlmdGVkID0gdGhpcy5pc1ZhbGlkKCkgJiZcbiAgICAgIGNvbXBhcmVBcnJheXMoYy5fYSwgb3RoZXIudG9BcnJheSgpKSA+IDA7XG4gIH0gZWxzZSB7XG4gICAgdGhpcy5faXNEU1RTaGlmdGVkID0gZmFsc2U7XG4gIH1cblxuICByZXR1cm4gdGhpcy5faXNEU1RTaGlmdGVkO1xufSovXG5cbi8vIGluIEtocm9ub3Ncbi8qZXhwb3J0IGZ1bmN0aW9uIGlzTG9jYWwoKSB7XG4gIHJldHVybiB0aGlzLmlzVmFsaWQoKSA/ICF0aGlzLl9pc1VUQyA6IGZhbHNlO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gaXNVdGNPZmZzZXQoKSB7XG4gIHJldHVybiB0aGlzLmlzVmFsaWQoKSA/IHRoaXMuX2lzVVRDIDogZmFsc2U7XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBpc1V0YygpIHtcbiAgcmV0dXJuIHRoaXMuaXNWYWxpZCgpID8gdGhpcy5faXNVVEMgJiYgdGhpcy5fb2Zmc2V0ID09PSAwIDogZmFsc2U7XG59Ki9cbiJdfQ==