/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { ChangeDetectionStrategy } from '../change_detection/constants';
import { compileComponent, compileDirective } from '../render3/jit/directive';
import { compilePipe } from '../render3/jit/pipe';
import { makeDecorator, makePropDecorator } from '../util/decorators';
/**
 * Type of the Directive metadata.
 *
 * @publicApi
 */
export const Directive = makeDecorator('Directive', (dir = {}) => dir, undefined, undefined, (type, meta) => compileDirective(type, meta));
/**
 * Component decorator and metadata.
 *
 * @Annotation
 * @publicApi
 */
export const Component = makeDecorator('Component', (c = {}) => ({ changeDetection: ChangeDetectionStrategy.Default, ...c }), Directive, undefined, (type, meta) => compileComponent(type, meta));
/**
 * @Annotation
 * @publicApi
 */
export const Pipe = makeDecorator('Pipe', (p) => ({ pure: true, ...p }), undefined, undefined, (type, meta) => compilePipe(type, meta));
/**
 * @Annotation
 * @publicApi
 */
export const Input = makePropDecorator('Input', (bindingPropertyName) => ({ bindingPropertyName }));
/**
 * @Annotation
 * @publicApi
 */
export const Output = makePropDecorator('Output', (bindingPropertyName) => ({ bindingPropertyName }));
/**
 * @Annotation
 * @publicApi
 */
export const HostBinding = makePropDecorator('HostBinding', (hostPropertyName) => ({ hostPropertyName }));
/**
 * Decorator that binds a DOM event to a host listener and supplies configuration metadata.
 * Angular invokes the supplied handler method when the host element emits the specified event,
 * and updates the bound element with the result.
 *
 * If the handler method returns false, applies `preventDefault` on the bound element.
 *
 * @usageNotes
 *
 * The following example declares a directive
 * that attaches a click listener to a button and counts clicks.
 *
 * ```ts
 * @Directive({selector: 'button[counting]'})
 * class CountClicks {
 *   numberOfClicks = 0;
 *
 *   @HostListener('click', ['$event.target'])
 *   onClick(btn) {
 *     console.log('button', btn, 'number of clicks:', this.numberOfClicks++);
 *   }
 * }
 *
 * @Component({
 *   selector: 'app',
 *   template: '<button counting>Increment</button>',
 * })
 * class App {}
 *
 * ```
 *
 * The following example registers another DOM event handler that listens for `Enter` key-press
 * events on the global `window`.
 * ``` ts
 * import { HostListener, Component } from "@angular/core";
 *
 * @Component({
 *   selector: 'app',
 *   template: `<h1>Hello, you have pressed enter {{counter}} number of times!</h1> Press enter key
 * to increment the counter.
 *   <button (click)="resetCounter()">Reset Counter</button>`
 * })
 * class AppComponent {
 *   counter = 0;
 *   @HostListener('window:keydown.enter', ['$event'])
 *   handleKeyDown(event: KeyboardEvent) {
 *     this.counter++;
 *   }
 *   resetCounter() {
 *     this.counter = 0;
 *   }
 * }
 * ```
 * The list of valid key names for `keydown` and `keyup` events
 * can be found here:
 * https://www.w3.org/TR/DOM-Level-3-Events-key/#named-key-attribute-values
 *
 * Note that keys can also be combined, e.g. `@HostListener('keydown.shift.a')`.
 *
 * The global target names that can be used to prefix an event name are
 * `document:`, `window:` and `body:`.
 *
 * @Annotation
 * @publicApi
 */
export const HostListener = makePropDecorator('HostListener', (eventName, args) => ({ eventName, args }));
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZGlyZWN0aXZlcy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uLy4uLy4uLy4uLy4uL3BhY2thZ2VzL2NvcmUvc3JjL21ldGFkYXRhL2RpcmVjdGl2ZXMudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQUE7Ozs7OztHQU1HO0FBRUgsT0FBTyxFQUFDLHVCQUF1QixFQUFDLE1BQU0sK0JBQStCLENBQUM7QUFHdEUsT0FBTyxFQUFDLGdCQUFnQixFQUFFLGdCQUFnQixFQUFDLE1BQU0sMEJBQTBCLENBQUM7QUFDNUUsT0FBTyxFQUFDLFdBQVcsRUFBQyxNQUFNLHFCQUFxQixDQUFDO0FBQ2hELE9BQU8sRUFBQyxhQUFhLEVBQUUsaUJBQWlCLEVBQWdCLE1BQU0sb0JBQW9CLENBQUM7QUFtVW5GOzs7O0dBSUc7QUFDSCxNQUFNLENBQUMsTUFBTSxTQUFTLEdBQXVCLGFBQWEsQ0FDdEQsV0FBVyxFQUFFLENBQUMsTUFBaUIsRUFBRSxFQUFFLEVBQUUsQ0FBQyxHQUFHLEVBQUUsU0FBUyxFQUFFLFNBQVMsRUFDL0QsQ0FBQyxJQUFlLEVBQUUsSUFBZSxFQUFFLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLENBQUMsQ0FBQztBQTBTeEU7Ozs7O0dBS0c7QUFDSCxNQUFNLENBQUMsTUFBTSxTQUFTLEdBQXVCLGFBQWEsQ0FDdEQsV0FBVyxFQUFFLENBQUMsSUFBZSxFQUFFLEVBQUUsRUFBRSxDQUFDLENBQUMsRUFBQyxlQUFlLEVBQUUsdUJBQXVCLENBQUMsT0FBTyxFQUFFLEdBQUcsQ0FBQyxFQUFDLENBQUMsRUFDOUYsU0FBUyxFQUFFLFNBQVMsRUFBRSxDQUFDLElBQWUsRUFBRSxJQUFlLEVBQUUsRUFBRSxDQUFDLGdCQUFnQixDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsQ0FBQyxDQUFDO0FBd0U5Rjs7O0dBR0c7QUFDSCxNQUFNLENBQUMsTUFBTSxJQUFJLEdBQWtCLGFBQWEsQ0FDNUMsTUFBTSxFQUFFLENBQUMsQ0FBTyxFQUFFLEVBQUUsQ0FBQyxDQUFDLEVBQUMsSUFBSSxFQUFFLElBQUksRUFBRSxHQUFHLENBQUMsRUFBQyxDQUFDLEVBQUUsU0FBUyxFQUFFLFNBQVMsRUFDL0QsQ0FBQyxJQUFlLEVBQUUsSUFBVSxFQUFFLEVBQUUsQ0FBQyxXQUFXLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxDQUFDLENBQUM7QUFvRTlEOzs7R0FHRztBQUNILE1BQU0sQ0FBQyxNQUFNLEtBQUssR0FDZCxpQkFBaUIsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxtQkFBNEIsRUFBRSxFQUFFLENBQUMsQ0FBQyxFQUFDLG1CQUFtQixFQUFDLENBQUMsQ0FBQyxDQUFDO0FBd0MxRjs7O0dBR0c7QUFDSCxNQUFNLENBQUMsTUFBTSxNQUFNLEdBQ2YsaUJBQWlCLENBQUMsUUFBUSxFQUFFLENBQUMsbUJBQTRCLEVBQUUsRUFBRSxDQUFDLENBQUMsRUFBQyxtQkFBbUIsRUFBQyxDQUFDLENBQUMsQ0FBQztBQXVEM0Y7OztHQUdHO0FBQ0gsTUFBTSxDQUFDLE1BQU0sV0FBVyxHQUNwQixpQkFBaUIsQ0FBQyxhQUFhLEVBQUUsQ0FBQyxnQkFBeUIsRUFBRSxFQUFFLENBQUMsQ0FBQyxFQUFDLGdCQUFnQixFQUFDLENBQUMsQ0FBQyxDQUFDO0FBc0MxRjs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztHQWdFRztBQUNILE1BQU0sQ0FBQyxNQUFNLFlBQVksR0FDckIsaUJBQWlCLENBQUMsY0FBYyxFQUFFLENBQUMsU0FBa0IsRUFBRSxJQUFlLEVBQUUsRUFBRSxDQUFDLENBQUMsRUFBQyxTQUFTLEVBQUUsSUFBSSxFQUFDLENBQUMsQ0FBQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBAbGljZW5zZVxuICogQ29weXJpZ2h0IEdvb2dsZSBMTEMgQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqXG4gKiBVc2Ugb2YgdGhpcyBzb3VyY2UgY29kZSBpcyBnb3Zlcm5lZCBieSBhbiBNSVQtc3R5bGUgbGljZW5zZSB0aGF0IGNhbiBiZVxuICogZm91bmQgaW4gdGhlIExJQ0VOU0UgZmlsZSBhdCBodHRwczovL2FuZ3VsYXIuaW8vbGljZW5zZVxuICovXG5cbmltcG9ydCB7Q2hhbmdlRGV0ZWN0aW9uU3RyYXRlZ3l9IGZyb20gJy4uL2NoYW5nZV9kZXRlY3Rpb24vY29uc3RhbnRzJztcbmltcG9ydCB7UHJvdmlkZXJ9IGZyb20gJy4uL2RpL2ludGVyZmFjZS9wcm92aWRlcic7XG5pbXBvcnQge1R5cGV9IGZyb20gJy4uL2ludGVyZmFjZS90eXBlJztcbmltcG9ydCB7Y29tcGlsZUNvbXBvbmVudCwgY29tcGlsZURpcmVjdGl2ZX0gZnJvbSAnLi4vcmVuZGVyMy9qaXQvZGlyZWN0aXZlJztcbmltcG9ydCB7Y29tcGlsZVBpcGV9IGZyb20gJy4uL3JlbmRlcjMvaml0L3BpcGUnO1xuaW1wb3J0IHttYWtlRGVjb3JhdG9yLCBtYWtlUHJvcERlY29yYXRvciwgVHlwZURlY29yYXRvcn0gZnJvbSAnLi4vdXRpbC9kZWNvcmF0b3JzJztcblxuaW1wb3J0IHtTY2hlbWFNZXRhZGF0YX0gZnJvbSAnLi9zY2hlbWEnO1xuaW1wb3J0IHtWaWV3RW5jYXBzdWxhdGlvbn0gZnJvbSAnLi92aWV3JztcblxuXG5cbi8qKlxuICogVHlwZSBvZiB0aGUgRGlyZWN0aXZlIGRlY29yYXRvciAvIGNvbnN0cnVjdG9yIGZ1bmN0aW9uLlxuICogQHB1YmxpY0FwaVxuICovXG5leHBvcnQgaW50ZXJmYWNlIERpcmVjdGl2ZURlY29yYXRvciB7XG4gIC8qKlxuICAgKiBEZWNvcmF0b3IgdGhhdCBtYXJrcyBhIGNsYXNzIGFzIGFuIEFuZ3VsYXIgZGlyZWN0aXZlLlxuICAgKiBZb3UgY2FuIGRlZmluZSB5b3VyIG93biBkaXJlY3RpdmVzIHRvIGF0dGFjaCBjdXN0b20gYmVoYXZpb3IgdG8gZWxlbWVudHMgaW4gdGhlIERPTS5cbiAgICpcbiAgICogVGhlIG9wdGlvbnMgcHJvdmlkZSBjb25maWd1cmF0aW9uIG1ldGFkYXRhIHRoYXQgZGV0ZXJtaW5lc1xuICAgKiBob3cgdGhlIGRpcmVjdGl2ZSBzaG91bGQgYmUgcHJvY2Vzc2VkLCBpbnN0YW50aWF0ZWQgYW5kIHVzZWQgYXRcbiAgICogcnVudGltZS5cbiAgICpcbiAgICogRGlyZWN0aXZlIGNsYXNzZXMsIGxpa2UgY29tcG9uZW50IGNsYXNzZXMsIGNhbiBpbXBsZW1lbnRcbiAgICogW2xpZmUtY3ljbGUgaG9va3NdKGd1aWRlL2xpZmVjeWNsZS1ob29rcykgdG8gaW5mbHVlbmNlIHRoZWlyIGNvbmZpZ3VyYXRpb24gYW5kIGJlaGF2aW9yLlxuICAgKlxuICAgKlxuICAgKiBAdXNhZ2VOb3Rlc1xuICAgKiBUbyBkZWZpbmUgYSBkaXJlY3RpdmUsIG1hcmsgdGhlIGNsYXNzIHdpdGggdGhlIGRlY29yYXRvciBhbmQgcHJvdmlkZSBtZXRhZGF0YS5cbiAgICpcbiAgICogYGBgdHNcbiAgICogaW1wb3J0IHtEaXJlY3RpdmV9IGZyb20gJ0Bhbmd1bGFyL2NvcmUnO1xuICAgKlxuICAgKiBARGlyZWN0aXZlKHtcbiAgICogICBzZWxlY3RvcjogJ215LWRpcmVjdGl2ZScsXG4gICAqIH0pXG4gICAqIGV4cG9ydCBjbGFzcyBNeURpcmVjdGl2ZSB7XG4gICAqIC4uLlxuICAgKiB9XG4gICAqIGBgYFxuICAgKlxuICAgKiAjIyMgRGVjbGFyaW5nIGRpcmVjdGl2ZXNcbiAgICpcbiAgICogSW4gb3JkZXIgdG8gbWFrZSBhIGRpcmVjdGl2ZSBhdmFpbGFibGUgdG8gb3RoZXIgY29tcG9uZW50cyBpbiB5b3VyIGFwcGxpY2F0aW9uLCB5b3Ugc2hvdWxkIGRvXG4gICAqIG9uZSBvZiB0aGUgZm9sbG93aW5nOlxuICAgKiAgLSBlaXRoZXIgbWFyayB0aGUgZGlyZWN0aXZlIGFzIFtzdGFuZGFsb25lXShndWlkZS9zdGFuZGFsb25lLWNvbXBvbmVudHMpLFxuICAgKiAgLSBvciBkZWNsYXJlIGl0IGluIGFuIE5nTW9kdWxlIGJ5IGFkZGluZyBpdCB0byB0aGUgYGRlY2xhcmF0aW9uc2AgYW5kIGBleHBvcnRzYCBmaWVsZHMuXG4gICAqXG4gICAqICoqIE1hcmtpbmcgYSBkaXJlY3RpdmUgYXMgc3RhbmRhbG9uZSAqKlxuICAgKlxuICAgKiBZb3UgY2FuIGFkZCB0aGUgYHN0YW5kYWxvbmU6IHRydWVgIGZsYWcgdG8gdGhlIERpcmVjdGl2ZSBkZWNvcmF0b3IgbWV0YWRhdGEgdG8gZGVjbGFyZSBpdCBhc1xuICAgKiBbc3RhbmRhbG9uZV0oZ3VpZGUvc3RhbmRhbG9uZS1jb21wb25lbnRzKTpcbiAgICpcbiAgICogYGBgdHNcbiAgICogQERpcmVjdGl2ZSh7XG4gICAqICAgc3RhbmRhbG9uZTogdHJ1ZSxcbiAgICogICBzZWxlY3RvcjogJ215LWRpcmVjdGl2ZScsXG4gICAqIH0pXG4gICAqIGNsYXNzIE15RGlyZWN0aXZlIHt9XG4gICAqIGBgYFxuICAgKlxuICAgKiBXaGVuIG1hcmtpbmcgYSBkaXJlY3RpdmUgYXMgc3RhbmRhbG9uZSwgcGxlYXNlIG1ha2Ugc3VyZSB0aGF0IHRoZSBkaXJlY3RpdmUgaXMgbm90IGFscmVhZHlcbiAgICogZGVjbGFyZWQgaW4gYW4gTmdNb2R1bGUuXG4gICAqXG4gICAqXG4gICAqICoqIERlY2xhcmluZyBhIGRpcmVjdGl2ZSBpbiBhbiBOZ01vZHVsZSAqKlxuICAgKlxuICAgKiBBbm90aGVyIGFwcHJvYWNoIGlzIHRvIGRlY2xhcmUgYSBkaXJlY3RpdmUgaW4gYW4gTmdNb2R1bGU6XG4gICAqXG4gICAqIGBgYHRzXG4gICAqIEBEaXJlY3RpdmUoe1xuICAgKiAgIHNlbGVjdG9yOiAnbXktZGlyZWN0aXZlJyxcbiAgICogfSlcbiAgICogY2xhc3MgTXlEaXJlY3RpdmUge31cbiAgICpcbiAgICogQE5nTW9kdWxlKHtcbiAgICogICBkZWNsYXJhdGlvbnM6IFtNeURpcmVjdGl2ZSwgU29tZUNvbXBvbmVudF0sXG4gICAqICAgZXhwb3J0czogW015RGlyZWN0aXZlXSwgLy8gbWFraW5nIGl0IGF2YWlsYWJsZSBvdXRzaWRlIG9mIHRoaXMgbW9kdWxlXG4gICAqIH0pXG4gICAqIGNsYXNzIFNvbWVOZ01vZHVsZSB7fVxuICAgKiBgYGBcbiAgICpcbiAgICogV2hlbiBkZWNsYXJpbmcgYSBkaXJlY3RpdmUgaW4gYW4gTmdNb2R1bGUsIHBsZWFzZSBtYWtlIHN1cmUgdGhhdDpcbiAgICogIC0gdGhlIGRpcmVjdGl2ZSBpcyBkZWNsYXJlZCBpbiBleGFjdGx5IG9uZSBOZ01vZHVsZS5cbiAgICogIC0gdGhlIGRpcmVjdGl2ZSBpcyBub3Qgc3RhbmRhbG9uZS5cbiAgICogIC0geW91IGRvIG5vdCByZS1kZWNsYXJlIGEgZGlyZWN0aXZlIGltcG9ydGVkIGZyb20gYW5vdGhlciBtb2R1bGUuXG4gICAqICAtIHRoZSBkaXJlY3RpdmUgaXMgaW5jbHVkZWQgaW50byB0aGUgYGV4cG9ydHNgIGZpZWxkIGFzIHdlbGwgaWYgeW91IHdhbnQgdGhpcyBkaXJlY3RpdmUgdG8gYmVcbiAgICogICAgYWNjZXNzaWJsZSBmb3IgY29tcG9uZW50cyBvdXRzaWRlIG9mIHRoZSBOZ01vZHVsZS5cbiAgICpcbiAgICpcbiAgICogQEFubm90YXRpb25cbiAgICovXG4gIChvYmo/OiBEaXJlY3RpdmUpOiBUeXBlRGVjb3JhdG9yO1xuXG4gIC8qKlxuICAgKiBTZWUgdGhlIGBEaXJlY3RpdmVgIGRlY29yYXRvci5cbiAgICovXG4gIG5ldyhvYmo/OiBEaXJlY3RpdmUpOiBEaXJlY3RpdmU7XG59XG5cbi8qKlxuICogRGlyZWN0aXZlIGRlY29yYXRvciBhbmQgbWV0YWRhdGEuXG4gKlxuICogQEFubm90YXRpb25cbiAqIEBwdWJsaWNBcGlcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBEaXJlY3RpdmUge1xuICAvKipcbiAgICogVGhlIENTUyBzZWxlY3RvciB0aGF0IGlkZW50aWZpZXMgdGhpcyBkaXJlY3RpdmUgaW4gYSB0ZW1wbGF0ZVxuICAgKiBhbmQgdHJpZ2dlcnMgaW5zdGFudGlhdGlvbiBvZiB0aGUgZGlyZWN0aXZlLlxuICAgKlxuICAgKiBEZWNsYXJlIGFzIG9uZSBvZiB0aGUgZm9sbG93aW5nOlxuICAgKlxuICAgKiAtIGBlbGVtZW50LW5hbWVgOiBTZWxlY3QgYnkgZWxlbWVudCBuYW1lLlxuICAgKiAtIGAuY2xhc3NgOiBTZWxlY3QgYnkgY2xhc3MgbmFtZS5cbiAgICogLSBgW2F0dHJpYnV0ZV1gOiBTZWxlY3QgYnkgYXR0cmlidXRlIG5hbWUuXG4gICAqIC0gYFthdHRyaWJ1dGU9dmFsdWVdYDogU2VsZWN0IGJ5IGF0dHJpYnV0ZSBuYW1lIGFuZCB2YWx1ZS5cbiAgICogLSBgOm5vdChzdWJfc2VsZWN0b3IpYDogU2VsZWN0IG9ubHkgaWYgdGhlIGVsZW1lbnQgZG9lcyBub3QgbWF0Y2ggdGhlIGBzdWJfc2VsZWN0b3JgLlxuICAgKiAtIGBzZWxlY3RvcjEsIHNlbGVjdG9yMmA6IFNlbGVjdCBpZiBlaXRoZXIgYHNlbGVjdG9yMWAgb3IgYHNlbGVjdG9yMmAgbWF0Y2hlcy5cbiAgICpcbiAgICogQW5ndWxhciBvbmx5IGFsbG93cyBkaXJlY3RpdmVzIHRvIGFwcGx5IG9uIENTUyBzZWxlY3RvcnMgdGhhdCBkbyBub3QgY3Jvc3NcbiAgICogZWxlbWVudCBib3VuZGFyaWVzLlxuICAgKlxuICAgKiBGb3IgdGhlIGZvbGxvd2luZyB0ZW1wbGF0ZSBIVE1MLCBhIGRpcmVjdGl2ZSB3aXRoIGFuIGBpbnB1dFt0eXBlPXRleHRdYCBzZWxlY3RvcixcbiAgICogd291bGQgYmUgaW5zdGFudGlhdGVkIG9ubHkgb24gdGhlIGA8aW5wdXQgdHlwZT1cInRleHRcIj5gIGVsZW1lbnQuXG4gICAqXG4gICAqIGBgYGh0bWxcbiAgICogPGZvcm0+XG4gICAqICAgPGlucHV0IHR5cGU9XCJ0ZXh0XCI+XG4gICAqICAgPGlucHV0IHR5cGU9XCJyYWRpb1wiPlxuICAgKiA8Zm9ybT5cbiAgICogYGBgXG4gICAqXG4gICAqL1xuICBzZWxlY3Rvcj86IHN0cmluZztcblxuICAvKipcbiAgICogRW51bWVyYXRlcyB0aGUgc2V0IG9mIGRhdGEtYm91bmQgaW5wdXQgcHJvcGVydGllcyBmb3IgYSBkaXJlY3RpdmVcbiAgICpcbiAgICogQW5ndWxhciBhdXRvbWF0aWNhbGx5IHVwZGF0ZXMgaW5wdXQgcHJvcGVydGllcyBkdXJpbmcgY2hhbmdlIGRldGVjdGlvbi5cbiAgICogVGhlIGBpbnB1dHNgIHByb3BlcnR5IGRlZmluZXMgYSBzZXQgb2YgYGRpcmVjdGl2ZVByb3BlcnR5YCB0byBgYmluZGluZ1Byb3BlcnR5YFxuICAgKiBjb25maWd1cmF0aW9uOlxuICAgKlxuICAgKiAtIGBkaXJlY3RpdmVQcm9wZXJ0eWAgc3BlY2lmaWVzIHRoZSBjb21wb25lbnQgcHJvcGVydHkgd2hlcmUgdGhlIHZhbHVlIGlzIHdyaXR0ZW4uXG4gICAqIC0gYGJpbmRpbmdQcm9wZXJ0eWAgc3BlY2lmaWVzIHRoZSBET00gcHJvcGVydHkgd2hlcmUgdGhlIHZhbHVlIGlzIHJlYWQgZnJvbS5cbiAgICpcbiAgICogV2hlbiBgYmluZGluZ1Byb3BlcnR5YCBpcyBub3QgcHJvdmlkZWQsIGl0IGlzIGFzc3VtZWQgdG8gYmUgZXF1YWwgdG8gYGRpcmVjdGl2ZVByb3BlcnR5YC5cbiAgICpcbiAgICogQHVzYWdlTm90ZXNcbiAgICpcbiAgICogVGhlIGZvbGxvd2luZyBleGFtcGxlIGNyZWF0ZXMgYSBjb21wb25lbnQgd2l0aCB0d28gZGF0YS1ib3VuZCBwcm9wZXJ0aWVzLlxuICAgKlxuICAgKiBgYGB0eXBlc2NyaXB0XG4gICAqIEBDb21wb25lbnQoe1xuICAgKiAgIHNlbGVjdG9yOiAnYmFuay1hY2NvdW50JyxcbiAgICogICBpbnB1dHM6IFsnYmFua05hbWUnLCAnaWQ6IGFjY291bnQtaWQnXSxcbiAgICogICB0ZW1wbGF0ZTogYFxuICAgKiAgICAgQmFuayBOYW1lOiB7e2JhbmtOYW1lfX1cbiAgICogICAgIEFjY291bnQgSWQ6IHt7aWR9fVxuICAgKiAgIGBcbiAgICogfSlcbiAgICogY2xhc3MgQmFua0FjY291bnQge1xuICAgKiAgIGJhbmtOYW1lOiBzdHJpbmc7XG4gICAqICAgaWQ6IHN0cmluZztcbiAgICogfVxuICAgKiBgYGBcbiAgICpcbiAgICovXG4gIGlucHV0cz86IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBFbnVtZXJhdGVzIHRoZSBzZXQgb2YgZXZlbnQtYm91bmQgb3V0cHV0IHByb3BlcnRpZXMuXG4gICAqXG4gICAqIFdoZW4gYW4gb3V0cHV0IHByb3BlcnR5IGVtaXRzIGFuIGV2ZW50LCBhbiBldmVudCBoYW5kbGVyIGF0dGFjaGVkIHRvIHRoYXQgZXZlbnRcbiAgICogaW4gdGhlIHRlbXBsYXRlIGlzIGludm9rZWQuXG4gICAqXG4gICAqIFRoZSBgb3V0cHV0c2AgcHJvcGVydHkgZGVmaW5lcyBhIHNldCBvZiBgZGlyZWN0aXZlUHJvcGVydHlgIHRvIGBiaW5kaW5nUHJvcGVydHlgXG4gICAqIGNvbmZpZ3VyYXRpb246XG4gICAqXG4gICAqIC0gYGRpcmVjdGl2ZVByb3BlcnR5YCBzcGVjaWZpZXMgdGhlIGNvbXBvbmVudCBwcm9wZXJ0eSB0aGF0IGVtaXRzIGV2ZW50cy5cbiAgICogLSBgYmluZGluZ1Byb3BlcnR5YCBzcGVjaWZpZXMgdGhlIERPTSBwcm9wZXJ0eSB0aGUgZXZlbnQgaGFuZGxlciBpcyBhdHRhY2hlZCB0by5cbiAgICpcbiAgICogQHVzYWdlTm90ZXNcbiAgICpcbiAgICogYGBgdHlwZXNjcmlwdFxuICAgKiBAQ29tcG9uZW50KHtcbiAgICogICBzZWxlY3RvcjogJ2NoaWxkLWRpcicsXG4gICAqICAgb3V0cHV0czogWyAnYmFua05hbWVDaGFuZ2UnIF1cbiAgICogICB0ZW1wbGF0ZTogYDxpbnB1dCAoaW5wdXQpPVwiYmFua05hbWVDaGFuZ2UuZW1pdCgkZXZlbnQudGFyZ2V0LnZhbHVlKVwiIC8+YFxuICAgKiB9KVxuICAgKiBjbGFzcyBDaGlsZERpciB7XG4gICAqICBiYW5rTmFtZUNoYW5nZTogRXZlbnRFbWl0dGVyPHN0cmluZz4gPSBuZXcgRXZlbnRFbWl0dGVyPHN0cmluZz4oKTtcbiAgICogfVxuICAgKlxuICAgKiBAQ29tcG9uZW50KHtcbiAgICogICBzZWxlY3RvcjogJ21haW4nLFxuICAgKiAgIHRlbXBsYXRlOiBgXG4gICAqICAgICB7eyBiYW5rTmFtZSB9fSA8Y2hpbGQtZGlyIChiYW5rTmFtZUNoYW5nZSk9XCJvbkJhbmtOYW1lQ2hhbmdlKCRldmVudClcIj48L2NoaWxkLWRpcj5cbiAgICogICBgXG4gICAqIH0pXG4gICAqIGNsYXNzIE1haW5Db21wb25lbnQge1xuICAgKiAgYmFua05hbWU6IHN0cmluZztcbiAgICpcbiAgICogICBvbkJhbmtOYW1lQ2hhbmdlKGJhbmtOYW1lOiBzdHJpbmcpIHtcbiAgICogICAgIHRoaXMuYmFua05hbWUgPSBiYW5rTmFtZTtcbiAgICogICB9XG4gICAqIH1cbiAgICogYGBgXG4gICAqXG4gICAqL1xuICBvdXRwdXRzPzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIENvbmZpZ3VyZXMgdGhlIFtpbmplY3Rvcl0oZ3VpZGUvZ2xvc3NhcnkjaW5qZWN0b3IpIG9mIHRoaXNcbiAgICogZGlyZWN0aXZlIG9yIGNvbXBvbmVudCB3aXRoIGEgW3Rva2VuXShndWlkZS9nbG9zc2FyeSNkaS10b2tlbilcbiAgICogdGhhdCBtYXBzIHRvIGEgW3Byb3ZpZGVyXShndWlkZS9nbG9zc2FyeSNwcm92aWRlcikgb2YgYSBkZXBlbmRlbmN5LlxuICAgKi9cbiAgcHJvdmlkZXJzPzogUHJvdmlkZXJbXTtcblxuICAvKipcbiAgICogRGVmaW5lcyB0aGUgbmFtZSB0aGF0IGNhbiBiZSB1c2VkIGluIHRoZSB0ZW1wbGF0ZSB0byBhc3NpZ24gdGhpcyBkaXJlY3RpdmUgdG8gYSB2YXJpYWJsZS5cbiAgICpcbiAgICogQHVzYWdlTm90ZXNcbiAgICpcbiAgICogYGBgdHNcbiAgICogQERpcmVjdGl2ZSh7XG4gICAqICAgc2VsZWN0b3I6ICdjaGlsZC1kaXInLFxuICAgKiAgIGV4cG9ydEFzOiAnY2hpbGQnXG4gICAqIH0pXG4gICAqIGNsYXNzIENoaWxkRGlyIHtcbiAgICogfVxuICAgKlxuICAgKiBAQ29tcG9uZW50KHtcbiAgICogICBzZWxlY3RvcjogJ21haW4nLFxuICAgKiAgIHRlbXBsYXRlOiBgPGNoaWxkLWRpciAjYz1cImNoaWxkXCI+PC9jaGlsZC1kaXI+YFxuICAgKiB9KVxuICAgKiBjbGFzcyBNYWluQ29tcG9uZW50IHtcbiAgICogfVxuICAgKiBgYGBcbiAgICpcbiAgICovXG4gIGV4cG9ydEFzPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBDb25maWd1cmVzIHRoZSBxdWVyaWVzIHRoYXQgd2lsbCBiZSBpbmplY3RlZCBpbnRvIHRoZSBkaXJlY3RpdmUuXG4gICAqXG4gICAqIENvbnRlbnQgcXVlcmllcyBhcmUgc2V0IGJlZm9yZSB0aGUgYG5nQWZ0ZXJDb250ZW50SW5pdGAgY2FsbGJhY2sgaXMgY2FsbGVkLlxuICAgKiBWaWV3IHF1ZXJpZXMgYXJlIHNldCBiZWZvcmUgdGhlIGBuZ0FmdGVyVmlld0luaXRgIGNhbGxiYWNrIGlzIGNhbGxlZC5cbiAgICpcbiAgICogQHVzYWdlTm90ZXNcbiAgICpcbiAgICogVGhlIGZvbGxvd2luZyBleGFtcGxlIHNob3dzIGhvdyBxdWVyaWVzIGFyZSBkZWZpbmVkXG4gICAqIGFuZCB3aGVuIHRoZWlyIHJlc3VsdHMgYXJlIGF2YWlsYWJsZSBpbiBsaWZlY3ljbGUgaG9va3M6XG4gICAqXG4gICAqIGBgYHRzXG4gICAqIEBDb21wb25lbnQoe1xuICAgKiAgIHNlbGVjdG9yOiAnc29tZURpcicsXG4gICAqICAgcXVlcmllczoge1xuICAgKiAgICAgY29udGVudENoaWxkcmVuOiBuZXcgQ29udGVudENoaWxkcmVuKENoaWxkRGlyZWN0aXZlKSxcbiAgICogICAgIHZpZXdDaGlsZHJlbjogbmV3IFZpZXdDaGlsZHJlbihDaGlsZERpcmVjdGl2ZSlcbiAgICogICB9LFxuICAgKiAgIHRlbXBsYXRlOiAnPGNoaWxkLWRpcmVjdGl2ZT48L2NoaWxkLWRpcmVjdGl2ZT4nXG4gICAqIH0pXG4gICAqIGNsYXNzIFNvbWVEaXIge1xuICAgKiAgIGNvbnRlbnRDaGlsZHJlbjogUXVlcnlMaXN0PENoaWxkRGlyZWN0aXZlPixcbiAgICogICB2aWV3Q2hpbGRyZW46IFF1ZXJ5TGlzdDxDaGlsZERpcmVjdGl2ZT5cbiAgICpcbiAgICogICBuZ0FmdGVyQ29udGVudEluaXQoKSB7XG4gICAqICAgICAvLyBjb250ZW50Q2hpbGRyZW4gaXMgc2V0XG4gICAqICAgfVxuICAgKlxuICAgKiAgIG5nQWZ0ZXJWaWV3SW5pdCgpIHtcbiAgICogICAgIC8vIHZpZXdDaGlsZHJlbiBpcyBzZXRcbiAgICogICB9XG4gICAqIH1cbiAgICogYGBgXG4gICAqXG4gICAqIEBBbm5vdGF0aW9uXG4gICAqL1xuICBxdWVyaWVzPzoge1trZXk6IHN0cmluZ106IGFueX07XG5cbiAgLyoqXG4gICAqIE1hcHMgY2xhc3MgcHJvcGVydGllcyB0byBob3N0IGVsZW1lbnQgYmluZGluZ3MgZm9yIHByb3BlcnRpZXMsXG4gICAqIGF0dHJpYnV0ZXMsIGFuZCBldmVudHMsIHVzaW5nIGEgc2V0IG9mIGtleS12YWx1ZSBwYWlycy5cbiAgICpcbiAgICogQW5ndWxhciBhdXRvbWF0aWNhbGx5IGNoZWNrcyBob3N0IHByb3BlcnR5IGJpbmRpbmdzIGR1cmluZyBjaGFuZ2UgZGV0ZWN0aW9uLlxuICAgKiBJZiBhIGJpbmRpbmcgY2hhbmdlcywgQW5ndWxhciB1cGRhdGVzIHRoZSBkaXJlY3RpdmUncyBob3N0IGVsZW1lbnQuXG4gICAqXG4gICAqIFdoZW4gdGhlIGtleSBpcyBhIHByb3BlcnR5IG9mIHRoZSBob3N0IGVsZW1lbnQsIHRoZSBwcm9wZXJ0eSB2YWx1ZSBpc1xuICAgKiB0aGUgcHJvcGFnYXRlZCB0byB0aGUgc3BlY2lmaWVkIERPTSBwcm9wZXJ0eS5cbiAgICpcbiAgICogV2hlbiB0aGUga2V5IGlzIGEgc3RhdGljIGF0dHJpYnV0ZSBpbiB0aGUgRE9NLCB0aGUgYXR0cmlidXRlIHZhbHVlXG4gICAqIGlzIHByb3BhZ2F0ZWQgdG8gdGhlIHNwZWNpZmllZCBwcm9wZXJ0eSBpbiB0aGUgaG9zdCBlbGVtZW50LlxuICAgKlxuICAgKiBGb3IgZXZlbnQgaGFuZGxpbmc6XG4gICAqIC0gVGhlIGtleSBpcyB0aGUgRE9NIGV2ZW50IHRoYXQgdGhlIGRpcmVjdGl2ZSBsaXN0ZW5zIHRvLlxuICAgKiBUbyBsaXN0ZW4gdG8gZ2xvYmFsIGV2ZW50cywgYWRkIHRoZSB0YXJnZXQgdG8gdGhlIGV2ZW50IG5hbWUuXG4gICAqIFRoZSB0YXJnZXQgY2FuIGJlIGB3aW5kb3dgLCBgZG9jdW1lbnRgIG9yIGBib2R5YC5cbiAgICogLSBUaGUgdmFsdWUgaXMgdGhlIHN0YXRlbWVudCB0byBleGVjdXRlIHdoZW4gdGhlIGV2ZW50IG9jY3Vycy4gSWYgdGhlXG4gICAqIHN0YXRlbWVudCBldmFsdWF0ZXMgdG8gYGZhbHNlYCwgdGhlbiBgcHJldmVudERlZmF1bHRgIGlzIGFwcGxpZWQgb24gdGhlIERPTVxuICAgKiBldmVudC4gQSBoYW5kbGVyIG1ldGhvZCBjYW4gcmVmZXIgdG8gdGhlIGAkZXZlbnRgIGxvY2FsIHZhcmlhYmxlLlxuICAgKlxuICAgKi9cbiAgaG9zdD86IHtba2V5OiBzdHJpbmddOiBzdHJpbmd9O1xuXG4gIC8qKlxuICAgKiBXaGVuIHByZXNlbnQsIHRoaXMgZGlyZWN0aXZlL2NvbXBvbmVudCBpcyBpZ25vcmVkIGJ5IHRoZSBBT1QgY29tcGlsZXIuXG4gICAqIEl0IHJlbWFpbnMgaW4gZGlzdHJpYnV0ZWQgY29kZSwgYW5kIHRoZSBKSVQgY29tcGlsZXIgYXR0ZW1wdHMgdG8gY29tcGlsZSBpdFxuICAgKiBhdCBydW4gdGltZSwgaW4gdGhlIGJyb3dzZXIuXG4gICAqIFRvIGVuc3VyZSB0aGUgY29ycmVjdCBiZWhhdmlvciwgdGhlIGFwcCBtdXN0IGltcG9ydCBgQGFuZ3VsYXIvY29tcGlsZXJgLlxuICAgKi9cbiAgaml0PzogdHJ1ZTtcblxuICAvKipcbiAgICogQW5ndWxhciBkaXJlY3RpdmVzIG1hcmtlZCBhcyBgc3RhbmRhbG9uZWAgZG8gbm90IG5lZWQgdG8gYmUgZGVjbGFyZWQgaW4gYW4gTmdNb2R1bGUuIFN1Y2hcbiAgICogZGlyZWN0aXZlcyBkb24ndCBkZXBlbmQgb24gYW55IFwiaW50ZXJtZWRpYXRlIGNvbnRleHRcIiBvZiBhbiBOZ01vZHVsZSAoZXguIGNvbmZpZ3VyZWRcbiAgICogcHJvdmlkZXJzKS5cbiAgICpcbiAgICogTW9yZSBpbmZvcm1hdGlvbiBhYm91dCBzdGFuZGFsb25lIGNvbXBvbmVudHMsIGRpcmVjdGl2ZXMgYW5kIHBpcGVzIGNhbiBiZSBmb3VuZCBpbiBbdGhpc1xuICAgKiBndWlkZV0oZ3VpZGUvc3RhbmRhbG9uZS1jb21wb25lbnRzKS5cbiAgICpcbiAgICogQGRldmVsb3BlclByZXZpZXdcbiAgICovXG4gIHN0YW5kYWxvbmU/OiBib29sZWFuO1xufVxuXG4vKipcbiAqIFR5cGUgb2YgdGhlIERpcmVjdGl2ZSBtZXRhZGF0YS5cbiAqXG4gKiBAcHVibGljQXBpXG4gKi9cbmV4cG9ydCBjb25zdCBEaXJlY3RpdmU6IERpcmVjdGl2ZURlY29yYXRvciA9IG1ha2VEZWNvcmF0b3IoXG4gICAgJ0RpcmVjdGl2ZScsIChkaXI6IERpcmVjdGl2ZSA9IHt9KSA9PiBkaXIsIHVuZGVmaW5lZCwgdW5kZWZpbmVkLFxuICAgICh0eXBlOiBUeXBlPGFueT4sIG1ldGE6IERpcmVjdGl2ZSkgPT4gY29tcGlsZURpcmVjdGl2ZSh0eXBlLCBtZXRhKSk7XG5cbi8qKlxuICogQ29tcG9uZW50IGRlY29yYXRvciBpbnRlcmZhY2VcbiAqXG4gKiBAcHVibGljQXBpXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQ29tcG9uZW50RGVjb3JhdG9yIHtcbiAgLyoqXG4gICAqIERlY29yYXRvciB0aGF0IG1hcmtzIGEgY2xhc3MgYXMgYW4gQW5ndWxhciBjb21wb25lbnQgYW5kIHByb3ZpZGVzIGNvbmZpZ3VyYXRpb25cbiAgICogbWV0YWRhdGEgdGhhdCBkZXRlcm1pbmVzIGhvdyB0aGUgY29tcG9uZW50IHNob3VsZCBiZSBwcm9jZXNzZWQsXG4gICAqIGluc3RhbnRpYXRlZCwgYW5kIHVzZWQgYXQgcnVudGltZS5cbiAgICpcbiAgICogQ29tcG9uZW50cyBhcmUgdGhlIG1vc3QgYmFzaWMgVUkgYnVpbGRpbmcgYmxvY2sgb2YgYW4gQW5ndWxhciBhcHAuXG4gICAqIEFuIEFuZ3VsYXIgYXBwIGNvbnRhaW5zIGEgdHJlZSBvZiBBbmd1bGFyIGNvbXBvbmVudHMuXG4gICAqXG4gICAqIEFuZ3VsYXIgY29tcG9uZW50cyBhcmUgYSBzdWJzZXQgb2YgZGlyZWN0aXZlcywgYWx3YXlzIGFzc29jaWF0ZWQgd2l0aCBhIHRlbXBsYXRlLlxuICAgKiBVbmxpa2Ugb3RoZXIgZGlyZWN0aXZlcywgb25seSBvbmUgY29tcG9uZW50IGNhbiBiZSBpbnN0YW50aWF0ZWQgZm9yIGEgZ2l2ZW4gZWxlbWVudCBpbiBhXG4gICAqIHRlbXBsYXRlLlxuICAgKlxuICAgKiBBIGNvbXBvbmVudCBtdXN0IGJlbG9uZyB0byBhbiBOZ01vZHVsZSBpbiBvcmRlciBmb3IgaXQgdG8gYmUgYXZhaWxhYmxlXG4gICAqIHRvIGFub3RoZXIgY29tcG9uZW50IG9yIGFwcGxpY2F0aW9uLiBUbyBtYWtlIGl0IGEgbWVtYmVyIG9mIGFuIE5nTW9kdWxlLFxuICAgKiBsaXN0IGl0IGluIHRoZSBgZGVjbGFyYXRpb25zYCBmaWVsZCBvZiB0aGUgYE5nTW9kdWxlYCBtZXRhZGF0YS5cbiAgICpcbiAgICogTm90ZSB0aGF0LCBpbiBhZGRpdGlvbiB0byB0aGVzZSBvcHRpb25zIGZvciBjb25maWd1cmluZyBhIGRpcmVjdGl2ZSxcbiAgICogeW91IGNhbiBjb250cm9sIGEgY29tcG9uZW50J3MgcnVudGltZSBiZWhhdmlvciBieSBpbXBsZW1lbnRpbmdcbiAgICogbGlmZS1jeWNsZSBob29rcy4gRm9yIG1vcmUgaW5mb3JtYXRpb24sIHNlZSB0aGVcbiAgICogW0xpZmVjeWNsZSBIb29rc10oZ3VpZGUvbGlmZWN5Y2xlLWhvb2tzKSBndWlkZS5cbiAgICpcbiAgICogQHVzYWdlTm90ZXNcbiAgICpcbiAgICogIyMjIFNldHRpbmcgY29tcG9uZW50IGlucHV0c1xuICAgKlxuICAgKiBUaGUgZm9sbG93aW5nIGV4YW1wbGUgY3JlYXRlcyBhIGNvbXBvbmVudCB3aXRoIHR3byBkYXRhLWJvdW5kIHByb3BlcnRpZXMsXG4gICAqIHNwZWNpZmllZCBieSB0aGUgYGlucHV0c2AgdmFsdWUuXG4gICAqXG4gICAqIDxjb2RlLWV4YW1wbGUgcGF0aD1cImNvcmUvdHMvbWV0YWRhdGEvZGlyZWN0aXZlcy50c1wiIHJlZ2lvbj1cImNvbXBvbmVudC1pbnB1dFwiPjwvY29kZS1leGFtcGxlPlxuICAgKlxuICAgKlxuICAgKiAjIyMgU2V0dGluZyBjb21wb25lbnQgb3V0cHV0c1xuICAgKlxuICAgKiBUaGUgZm9sbG93aW5nIGV4YW1wbGUgc2hvd3MgdHdvIGV2ZW50IGVtaXR0ZXJzIHRoYXQgZW1pdCBvbiBhbiBpbnRlcnZhbC4gT25lXG4gICAqIGVtaXRzIGFuIG91dHB1dCBldmVyeSBzZWNvbmQsIHdoaWxlIHRoZSBvdGhlciBlbWl0cyBldmVyeSBmaXZlIHNlY29uZHMuXG4gICAqXG4gICAqIHtAZXhhbXBsZSBjb3JlL3RzL21ldGFkYXRhL2RpcmVjdGl2ZXMudHMgcmVnaW9uPSdjb21wb25lbnQtb3V0cHV0LWludGVydmFsJ31cbiAgICpcbiAgICogIyMjIEluamVjdGluZyBhIGNsYXNzIHdpdGggYSB2aWV3IHByb3ZpZGVyXG4gICAqXG4gICAqIFRoZSBmb2xsb3dpbmcgc2ltcGxlIGV4YW1wbGUgaW5qZWN0cyBhIGNsYXNzIGludG8gYSBjb21wb25lbnRcbiAgICogdXNpbmcgdGhlIHZpZXcgcHJvdmlkZXIgc3BlY2lmaWVkIGluIGNvbXBvbmVudCBtZXRhZGF0YTpcbiAgICpcbiAgICogYGBgdHNcbiAgICogY2xhc3MgR3JlZXRlciB7XG4gICAqICAgIGdyZWV0KG5hbWU6c3RyaW5nKSB7XG4gICAqICAgICAgcmV0dXJuICdIZWxsbyAnICsgbmFtZSArICchJztcbiAgICogICAgfVxuICAgKiB9XG4gICAqXG4gICAqIEBEaXJlY3RpdmUoe1xuICAgKiAgIHNlbGVjdG9yOiAnbmVlZHMtZ3JlZXRlcidcbiAgICogfSlcbiAgICogY2xhc3MgTmVlZHNHcmVldGVyIHtcbiAgICogICBncmVldGVyOkdyZWV0ZXI7XG4gICAqXG4gICAqICAgY29uc3RydWN0b3IoZ3JlZXRlcjpHcmVldGVyKSB7XG4gICAqICAgICB0aGlzLmdyZWV0ZXIgPSBncmVldGVyO1xuICAgKiAgIH1cbiAgICogfVxuICAgKlxuICAgKiBAQ29tcG9uZW50KHtcbiAgICogICBzZWxlY3RvcjogJ2dyZWV0JyxcbiAgICogICB2aWV3UHJvdmlkZXJzOiBbXG4gICAqICAgICBHcmVldGVyXG4gICAqICAgXSxcbiAgICogICB0ZW1wbGF0ZTogYDxuZWVkcy1ncmVldGVyPjwvbmVlZHMtZ3JlZXRlcj5gXG4gICAqIH0pXG4gICAqIGNsYXNzIEhlbGxvV29ybGQge1xuICAgKiB9XG4gICAqXG4gICAqIGBgYFxuICAgKlxuICAgKiAjIyMgUHJlc2VydmluZyB3aGl0ZXNwYWNlXG4gICAqXG4gICAqIFJlbW92aW5nIHdoaXRlc3BhY2UgY2FuIGdyZWF0bHkgcmVkdWNlIEFPVC1nZW5lcmF0ZWQgY29kZSBzaXplIGFuZCBzcGVlZCB1cCB2aWV3IGNyZWF0aW9uLlxuICAgKiBBcyBvZiBBbmd1bGFyIDYsIHRoZSBkZWZhdWx0IGZvciBgcHJlc2VydmVXaGl0ZXNwYWNlc2AgaXMgZmFsc2UgKHdoaXRlc3BhY2UgaXMgcmVtb3ZlZCkuXG4gICAqIFRvIGNoYW5nZSB0aGUgZGVmYXVsdCBzZXR0aW5nIGZvciBhbGwgY29tcG9uZW50cyBpbiB5b3VyIGFwcGxpY2F0aW9uLCBzZXRcbiAgICogdGhlIGBwcmVzZXJ2ZVdoaXRlc3BhY2VzYCBvcHRpb24gb2YgdGhlIEFPVCBjb21waWxlci5cbiAgICpcbiAgICogQnkgZGVmYXVsdCwgdGhlIEFPVCBjb21waWxlciByZW1vdmVzIHdoaXRlc3BhY2UgY2hhcmFjdGVycyBhcyBmb2xsb3dzOlxuICAgKiAqIFRyaW1zIGFsbCB3aGl0ZXNwYWNlcyBhdCB0aGUgYmVnaW5uaW5nIGFuZCB0aGUgZW5kIG9mIGEgdGVtcGxhdGUuXG4gICAqICogUmVtb3ZlcyB3aGl0ZXNwYWNlLW9ubHkgdGV4dCBub2Rlcy4gRm9yIGV4YW1wbGUsXG4gICAqXG4gICAqIGBgYGh0bWxcbiAgICogPGJ1dHRvbj5BY3Rpb24gMTwvYnV0dG9uPiAgPGJ1dHRvbj5BY3Rpb24gMjwvYnV0dG9uPlxuICAgKiBgYGBcbiAgICpcbiAgICogYmVjb21lczpcbiAgICpcbiAgICogYGBgaHRtbFxuICAgKiA8YnV0dG9uPkFjdGlvbiAxPC9idXR0b24+PGJ1dHRvbj5BY3Rpb24gMjwvYnV0dG9uPlxuICAgKiBgYGBcbiAgICpcbiAgICogKiBSZXBsYWNlcyBhIHNlcmllcyBvZiB3aGl0ZXNwYWNlIGNoYXJhY3RlcnMgaW4gdGV4dCBub2RlcyB3aXRoIGEgc2luZ2xlIHNwYWNlLlxuICAgKiBGb3IgZXhhbXBsZSwgYDxzcGFuPlxcbiBzb21lIHRleHRcXG48L3NwYW4+YCBiZWNvbWVzIGA8c3Bhbj4gc29tZSB0ZXh0IDwvc3Bhbj5gLlxuICAgKiAqIERvZXMgTk9UIGFsdGVyIHRleHQgbm9kZXMgaW5zaWRlIEhUTUwgdGFncyBzdWNoIGFzIGA8cHJlPmAgb3IgYDx0ZXh0YXJlYT5gLFxuICAgKiB3aGVyZSB3aGl0ZXNwYWNlIGNoYXJhY3RlcnMgYXJlIHNpZ25pZmljYW50LlxuICAgKlxuICAgKiBOb3RlIHRoYXQgdGhlc2UgdHJhbnNmb3JtYXRpb25zIGNhbiBpbmZsdWVuY2UgRE9NIG5vZGVzIGxheW91dCwgYWx0aG91Z2ggaW1wYWN0XG4gICAqIHNob3VsZCBiZSBtaW5pbWFsLlxuICAgKlxuICAgKiBZb3UgY2FuIG92ZXJyaWRlIHRoZSBkZWZhdWx0IGJlaGF2aW9yIHRvIHByZXNlcnZlIHdoaXRlc3BhY2UgY2hhcmFjdGVyc1xuICAgKiBpbiBjZXJ0YWluIGZyYWdtZW50cyBvZiBhIHRlbXBsYXRlLiBGb3IgZXhhbXBsZSwgeW91IGNhbiBleGNsdWRlIGFuIGVudGlyZVxuICAgKiBET00gc3ViLXRyZWUgYnkgdXNpbmcgdGhlIGBuZ1ByZXNlcnZlV2hpdGVzcGFjZXNgIGF0dHJpYnV0ZTpcbiAgICpcbiAgICogYGBgaHRtbFxuICAgKiA8ZGl2IG5nUHJlc2VydmVXaGl0ZXNwYWNlcz5cbiAgICogICAgIHdoaXRlc3BhY2VzIGFyZSBwcmVzZXJ2ZWQgaGVyZVxuICAgKiAgICAgPHNwYW4+ICAgIGFuZCBoZXJlIDwvc3Bhbj5cbiAgICogPC9kaXY+XG4gICAqIGBgYFxuICAgKlxuICAgKiBZb3UgY2FuIGZvcmNlIGEgc2luZ2xlIHNwYWNlIHRvIGJlIHByZXNlcnZlZCBpbiBhIHRleHQgbm9kZSBieSB1c2luZyBgJm5nc3A7YCxcbiAgICogd2hpY2ggaXMgcmVwbGFjZWQgd2l0aCBhIHNwYWNlIGNoYXJhY3RlciBieSBBbmd1bGFyJ3MgdGVtcGxhdGVcbiAgICogY29tcGlsZXI6XG4gICAqXG4gICAqIGBgYGh0bWxcbiAgICogPGE+U3BhY2VzPC9hPiZuZ3NwOzxhPmJldHdlZW48L2E+Jm5nc3A7PGE+bGlua3MuPC9hPlxuICAgKiA8IS0tIGNvbXBpbGVkIHRvIGJlIGVxdWl2YWxlbnQgdG86XG4gICAqICA8YT5TcGFjZXM8L2E+IDxhPmJldHdlZW48L2E+IDxhPmxpbmtzLjwvYT4gIC0tPlxuICAgKiBgYGBcbiAgICpcbiAgICogTm90ZSB0aGF0IHNlcXVlbmNlcyBvZiBgJm5nc3A7YCBhcmUgc3RpbGwgY29sbGFwc2VkIHRvIGp1c3Qgb25lIHNwYWNlIGNoYXJhY3RlciB3aGVuXG4gICAqIHRoZSBgcHJlc2VydmVXaGl0ZXNwYWNlc2Agb3B0aW9uIGlzIHNldCB0byBgZmFsc2VgLlxuICAgKlxuICAgKiBgYGBodG1sXG4gICAqIDxhPmJlZm9yZTwvYT4mbmdzcDsmbmdzcDsmbmdzcDs8YT5hZnRlcjwvYT5cbiAgICogPCEtLSBjb21waWxlZCB0byBiZSBlcXVpdmFsZW50IHRvOlxuICAgKiAgPGE+YmVmb3JlPC9hPiA8YT5hZnRlcjwvYT4gLS0+XG4gICAqIGBgYFxuICAgKlxuICAgKiBUbyBwcmVzZXJ2ZSBzZXF1ZW5jZXMgb2Ygd2hpdGVzcGFjZSBjaGFyYWN0ZXJzLCB1c2UgdGhlXG4gICAqIGBuZ1ByZXNlcnZlV2hpdGVzcGFjZXNgIGF0dHJpYnV0ZS5cbiAgICpcbiAgICogQEFubm90YXRpb25cbiAgICovXG4gIChvYmo6IENvbXBvbmVudCk6IFR5cGVEZWNvcmF0b3I7XG4gIC8qKlxuICAgKiBTZWUgdGhlIGBDb21wb25lbnRgIGRlY29yYXRvci5cbiAgICovXG4gIG5ldyhvYmo6IENvbXBvbmVudCk6IENvbXBvbmVudDtcbn1cblxuLyoqXG4gKiBTdXBwbGllcyBjb25maWd1cmF0aW9uIG1ldGFkYXRhIGZvciBhbiBBbmd1bGFyIGNvbXBvbmVudC5cbiAqXG4gKiBAcHVibGljQXBpXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQ29tcG9uZW50IGV4dGVuZHMgRGlyZWN0aXZlIHtcbiAgLyoqXG4gICAqIFRoZSBjaGFuZ2UtZGV0ZWN0aW9uIHN0cmF0ZWd5IHRvIHVzZSBmb3IgdGhpcyBjb21wb25lbnQuXG4gICAqXG4gICAqIFdoZW4gYSBjb21wb25lbnQgaXMgaW5zdGFudGlhdGVkLCBBbmd1bGFyIGNyZWF0ZXMgYSBjaGFuZ2UgZGV0ZWN0b3IsXG4gICAqIHdoaWNoIGlzIHJlc3BvbnNpYmxlIGZvciBwcm9wYWdhdGluZyB0aGUgY29tcG9uZW50J3MgYmluZGluZ3MuXG4gICAqIFRoZSBzdHJhdGVneSBpcyBvbmUgb2Y6XG4gICAqIC0gYENoYW5nZURldGVjdGlvblN0cmF0ZWd5I09uUHVzaGAgc2V0cyB0aGUgc3RyYXRlZ3kgdG8gYENoZWNrT25jZWAgKG9uIGRlbWFuZCkuXG4gICAqIC0gYENoYW5nZURldGVjdGlvblN0cmF0ZWd5I0RlZmF1bHRgIHNldHMgdGhlIHN0cmF0ZWd5IHRvIGBDaGVja0Fsd2F5c2AuXG4gICAqL1xuICBjaGFuZ2VEZXRlY3Rpb24/OiBDaGFuZ2VEZXRlY3Rpb25TdHJhdGVneTtcblxuICAvKipcbiAgICogRGVmaW5lcyB0aGUgc2V0IG9mIGluamVjdGFibGUgb2JqZWN0cyB0aGF0IGFyZSB2aXNpYmxlIHRvIGl0cyB2aWV3IERPTSBjaGlsZHJlbi5cbiAgICogU2VlIFtleGFtcGxlXSgjaW5qZWN0aW5nLWEtY2xhc3Mtd2l0aC1hLXZpZXctcHJvdmlkZXIpLlxuICAgKlxuICAgKi9cbiAgdmlld1Byb3ZpZGVycz86IFByb3ZpZGVyW107XG5cbiAgLyoqXG4gICAqIFRoZSBtb2R1bGUgSUQgb2YgdGhlIG1vZHVsZSB0aGF0IGNvbnRhaW5zIHRoZSBjb21wb25lbnQuXG4gICAqIFRoZSBjb21wb25lbnQgbXVzdCBiZSBhYmxlIHRvIHJlc29sdmUgcmVsYXRpdmUgVVJMcyBmb3IgdGVtcGxhdGVzIGFuZCBzdHlsZXMuXG4gICAqIFN5c3RlbUpTIGV4cG9zZXMgdGhlIGBfX21vZHVsZU5hbWVgIHZhcmlhYmxlIHdpdGhpbiBlYWNoIG1vZHVsZS5cbiAgICogSW4gQ29tbW9uSlMsIHRoaXMgY2FuICBiZSBzZXQgdG8gYG1vZHVsZS5pZGAuXG4gICAqXG4gICAqL1xuICBtb2R1bGVJZD86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIHJlbGF0aXZlIHBhdGggb3IgYWJzb2x1dGUgVVJMIG9mIGEgdGVtcGxhdGUgZmlsZSBmb3IgYW4gQW5ndWxhciBjb21wb25lbnQuXG4gICAqIElmIHByb3ZpZGVkLCBkbyBub3Qgc3VwcGx5IGFuIGlubGluZSB0ZW1wbGF0ZSB1c2luZyBgdGVtcGxhdGVgLlxuICAgKlxuICAgKi9cbiAgdGVtcGxhdGVVcmw/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEFuIGlubGluZSB0ZW1wbGF0ZSBmb3IgYW4gQW5ndWxhciBjb21wb25lbnQuIElmIHByb3ZpZGVkLFxuICAgKiBkbyBub3Qgc3VwcGx5IGEgdGVtcGxhdGUgZmlsZSB1c2luZyBgdGVtcGxhdGVVcmxgLlxuICAgKlxuICAgKi9cbiAgdGVtcGxhdGU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIE9uZSBvciBtb3JlIHJlbGF0aXZlIHBhdGhzIG9yIGFic29sdXRlIFVSTHMgZm9yIGZpbGVzIGNvbnRhaW5pbmcgQ1NTIHN0eWxlc2hlZXRzIHRvIHVzZVxuICAgKiBpbiB0aGlzIGNvbXBvbmVudC5cbiAgICovXG4gIHN0eWxlVXJscz86IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBPbmUgb3IgbW9yZSBpbmxpbmUgQ1NTIHN0eWxlc2hlZXRzIHRvIHVzZVxuICAgKiBpbiB0aGlzIGNvbXBvbmVudC5cbiAgICovXG4gIHN0eWxlcz86IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBPbmUgb3IgbW9yZSBhbmltYXRpb24gYHRyaWdnZXIoKWAgY2FsbHMsIGNvbnRhaW5pbmdcbiAgICogW2BzdGF0ZSgpYF0oYXBpL2FuaW1hdGlvbnMvc3RhdGUpIGFuZCBgdHJhbnNpdGlvbigpYCBkZWZpbml0aW9ucy5cbiAgICogU2VlIHRoZSBbQW5pbWF0aW9ucyBndWlkZV0oL2d1aWRlL2FuaW1hdGlvbnMpIGFuZCBhbmltYXRpb25zIEFQSSBkb2N1bWVudGF0aW9uLlxuICAgKlxuICAgKi9cbiAgYW5pbWF0aW9ucz86IGFueVtdO1xuXG4gIC8qKlxuICAgKiBBbiBlbmNhcHN1bGF0aW9uIHBvbGljeSBmb3IgdGhlIGNvbXBvbmVudCdzIHN0eWxpbmcuXG4gICAqIFBvc3NpYmxlIHZhbHVlczpcbiAgICogLSBgVmlld0VuY2Fwc3VsYXRpb24uRW11bGF0ZWRgOiBBcHBseSBtb2RpZmllZCBjb21wb25lbnQgc3R5bGVzIGluIG9yZGVyIHRvIGVtdWxhdGVcbiAgICogICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBhIG5hdGl2ZSBTaGFkb3cgRE9NIENTUyBlbmNhcHN1bGF0aW9uIGJlaGF2aW9yLlxuICAgKiAtIGBWaWV3RW5jYXBzdWxhdGlvbi5Ob25lYDogQXBwbHkgY29tcG9uZW50IHN0eWxlcyBnbG9iYWxseSB3aXRob3V0IGFueSBzb3J0IG9mIGVuY2Fwc3VsYXRpb24uXG4gICAqIC0gYFZpZXdFbmNhcHN1bGF0aW9uLlNoYWRvd0RvbWA6IFVzZSB0aGUgYnJvd3NlcidzIG5hdGl2ZSBTaGFkb3cgRE9NIEFQSSB0byBlbmNhcHN1bGF0ZSBzdHlsZXMuXG4gICAqXG4gICAqIElmIG5vdCBzdXBwbGllZCwgdGhlIHZhbHVlIGlzIHRha2VuIGZyb20gdGhlIGBDb21waWxlck9wdGlvbnNgXG4gICAqIHdoaWNoIGRlZmF1bHRzIHRvIGBWaWV3RW5jYXBzdWxhdGlvbi5FbXVsYXRlZGAuXG4gICAqXG4gICAqIElmIHRoZSBwb2xpY3kgaXMgYFZpZXdFbmNhcHN1bGF0aW9uLkVtdWxhdGVkYCBhbmQgdGhlIGNvbXBvbmVudCBoYXMgbm9cbiAgICoge0BsaW5rIENvbXBvbmVudCNzdHlsZXMgc3R5bGVzfSBub3Ige0BsaW5rIENvbXBvbmVudCNzdHlsZVVybHMgc3R5bGVVcmxzfSxcbiAgICogdGhlIHBvbGljeSBpcyBhdXRvbWF0aWNhbGx5IHN3aXRjaGVkIHRvIGBWaWV3RW5jYXBzdWxhdGlvbi5Ob25lYC5cbiAgICovXG4gIGVuY2Fwc3VsYXRpb24/OiBWaWV3RW5jYXBzdWxhdGlvbjtcblxuICAvKipcbiAgICogT3ZlcnJpZGVzIHRoZSBkZWZhdWx0IGludGVycG9sYXRpb24gc3RhcnQgYW5kIGVuZCBkZWxpbWl0ZXJzIChge3tgIGFuZCBgfX1gKS5cbiAgICovXG4gIGludGVycG9sYXRpb24/OiBbc3RyaW5nLCBzdHJpbmddO1xuXG4gIC8qKlxuICAgKiBBIHNldCBvZiBjb21wb25lbnRzIHRoYXQgc2hvdWxkIGJlIGNvbXBpbGVkIGFsb25nIHdpdGhcbiAgICogdGhpcyBjb21wb25lbnQuIEZvciBlYWNoIGNvbXBvbmVudCBsaXN0ZWQgaGVyZSxcbiAgICogQW5ndWxhciBjcmVhdGVzIGEge0BsaW5rIENvbXBvbmVudEZhY3Rvcnl9IGFuZCBzdG9yZXMgaXQgaW4gdGhlXG4gICAqIHtAbGluayBDb21wb25lbnRGYWN0b3J5UmVzb2x2ZXJ9LlxuICAgKiBAZGVwcmVjYXRlZCBTaW5jZSA5LjAuMC4gV2l0aCBJdnksIHRoaXMgcHJvcGVydHkgaXMgbm8gbG9uZ2VyIG5lY2Vzc2FyeS5cbiAgICovXG4gIGVudHJ5Q29tcG9uZW50cz86IEFycmF5PFR5cGU8YW55PnxhbnlbXT47XG5cbiAgLyoqXG4gICAqIFRydWUgdG8gcHJlc2VydmUgb3IgZmFsc2UgdG8gcmVtb3ZlIHBvdGVudGlhbGx5IHN1cGVyZmx1b3VzIHdoaXRlc3BhY2UgY2hhcmFjdGVyc1xuICAgKiBmcm9tIHRoZSBjb21waWxlZCB0ZW1wbGF0ZS4gV2hpdGVzcGFjZSBjaGFyYWN0ZXJzIGFyZSB0aG9zZSBtYXRjaGluZyB0aGUgYFxcc2BcbiAgICogY2hhcmFjdGVyIGNsYXNzIGluIEphdmFTY3JpcHQgcmVndWxhciBleHByZXNzaW9ucy4gRGVmYXVsdCBpcyBmYWxzZSwgdW5sZXNzXG4gICAqIG92ZXJyaWRkZW4gaW4gY29tcGlsZXIgb3B0aW9ucy5cbiAgICovXG4gIHByZXNlcnZlV2hpdGVzcGFjZXM/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBBbmd1bGFyIGNvbXBvbmVudHMgbWFya2VkIGFzIGBzdGFuZGFsb25lYCBkbyBub3QgbmVlZCB0byBiZSBkZWNsYXJlZCBpbiBhbiBOZ01vZHVsZS4gU3VjaFxuICAgKiBjb21wb25lbnRzIGRpcmVjdGx5IG1hbmFnZSB0aGVpciBvd24gdGVtcGxhdGUgZGVwZW5kZW5jaWVzIChjb21wb25lbnRzLCBkaXJlY3RpdmVzIGFuZCBwaXBlc1xuICAgKiB1c2VkIGluIGEgdGVtcGxhdGUpIHZpYSB0aGUgaW1wb3J0cyBwcm9wZXJ0eS5cbiAgICpcbiAgICogTW9yZSBpbmZvcm1hdGlvbiBhYm91dCBzdGFuZGFsb25lIGNvbXBvbmVudHMsIGRpcmVjdGl2ZXMgYW5kIHBpcGVzIGNhbiBiZSBmb3VuZCBpbiBbdGhpc1xuICAgKiBndWlkZV0oZ3VpZGUvc3RhbmRhbG9uZS1jb21wb25lbnRzKS5cbiAgICpcbiAgICogQGRldmVsb3BlclByZXZpZXdcbiAgICovXG4gIHN0YW5kYWxvbmU/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgaW1wb3J0cyBwcm9wZXJ0eSBzcGVjaWZpZXMgdGhlIHN0YW5kYWxvbmUgY29tcG9uZW50J3MgdGVtcGxhdGUgZGVwZW5kZW5jaWVzIOKAlCB0aG9zZVxuICAgKiBkaXJlY3RpdmVzLCBjb21wb25lbnRzLCBhbmQgcGlwZXMgdGhhdCBjYW4gYmUgdXNlZCB3aXRoaW4gaXRzIHRlbXBsYXRlLiBTdGFuZGFsb25lIGNvbXBvbmVudHNcbiAgICogY2FuIGltcG9ydCBvdGhlciBzdGFuZGFsb25lIGNvbXBvbmVudHMsIGRpcmVjdGl2ZXMgYW5kIHBpcGVzIGFzIHdlbGwgYXMgZXhpc3RpbmcgTmdNb2R1bGVzLlxuICAgKlxuICAgKiBUaGlzIHByb3BlcnR5IGlzIG9ubHkgYXZhaWxhYmxlIGZvciBzdGFuZGFsb25lIGNvbXBvbmVudHMgLSBzcGVjaWZ5aW5nIGl0IGZvciBjb21wb25lbnRzXG4gICAqIGRlY2xhcmVkIGluIGFuIE5nTW9kdWxlIGdlbmVyYXRlcyBhIGNvbXBpbGF0aW9uIGVycm9yLlxuICAgKlxuICAgKiBNb3JlIGluZm9ybWF0aW9uIGFib3V0IHN0YW5kYWxvbmUgY29tcG9uZW50cywgZGlyZWN0aXZlcyBhbmQgcGlwZXMgY2FuIGJlIGZvdW5kIGluIFt0aGlzXG4gICAqIGd1aWRlXShndWlkZS9zdGFuZGFsb25lLWNvbXBvbmVudHMpLlxuICAgKlxuICAgKiBAZGV2ZWxvcGVyUHJldmlld1xuICAgKi9cbiAgaW1wb3J0cz86IChUeXBlPGFueT58YW55W10pW107XG5cbiAgLyoqXG4gICAqIFRoZSBzZXQgb2Ygc2NoZW1hcyB0aGF0IGRlY2xhcmUgZWxlbWVudHMgdG8gYmUgYWxsb3dlZCBpbiBhIHN0YW5kYWxvbmUgY29tcG9uZW50LiBFbGVtZW50cyBhbmRcbiAgICogcHJvcGVydGllcyB0aGF0IGFyZSBuZWl0aGVyIEFuZ3VsYXIgY29tcG9uZW50cyBub3IgZGlyZWN0aXZlcyBtdXN0IGJlIGRlY2xhcmVkIGluIGEgc2NoZW1hLlxuICAgKlxuICAgKiBUaGlzIHByb3BlcnR5IGlzIG9ubHkgYXZhaWxhYmxlIGZvciBzdGFuZGFsb25lIGNvbXBvbmVudHMgLSBzcGVjaWZ5aW5nIGl0IGZvciBjb21wb25lbnRzXG4gICAqIGRlY2xhcmVkIGluIGFuIE5nTW9kdWxlIGdlbmVyYXRlcyBhIGNvbXBpbGF0aW9uIGVycm9yLlxuICAgKlxuICAgKiBNb3JlIGluZm9ybWF0aW9uIGFib3V0IHN0YW5kYWxvbmUgY29tcG9uZW50cywgZGlyZWN0aXZlcyBhbmQgcGlwZXMgY2FuIGJlIGZvdW5kIGluIFt0aGlzXG4gICAqIGd1aWRlXShndWlkZS9zdGFuZGFsb25lLWNvbXBvbmVudHMpLlxuICAgKi9cbiAgc2NoZW1hcz86IFNjaGVtYU1ldGFkYXRhW107XG59XG5cbi8qKlxuICogQ29tcG9uZW50IGRlY29yYXRvciBhbmQgbWV0YWRhdGEuXG4gKlxuICogQEFubm90YXRpb25cbiAqIEBwdWJsaWNBcGlcbiAqL1xuZXhwb3J0IGNvbnN0IENvbXBvbmVudDogQ29tcG9uZW50RGVjb3JhdG9yID0gbWFrZURlY29yYXRvcihcbiAgICAnQ29tcG9uZW50JywgKGM6IENvbXBvbmVudCA9IHt9KSA9PiAoe2NoYW5nZURldGVjdGlvbjogQ2hhbmdlRGV0ZWN0aW9uU3RyYXRlZ3kuRGVmYXVsdCwgLi4uY30pLFxuICAgIERpcmVjdGl2ZSwgdW5kZWZpbmVkLCAodHlwZTogVHlwZTxhbnk+LCBtZXRhOiBDb21wb25lbnQpID0+IGNvbXBpbGVDb21wb25lbnQodHlwZSwgbWV0YSkpO1xuXG4vKipcbiAqIFR5cGUgb2YgdGhlIFBpcGUgZGVjb3JhdG9yIC8gY29uc3RydWN0b3IgZnVuY3Rpb24uXG4gKlxuICogQHB1YmxpY0FwaVxuICovXG5leHBvcnQgaW50ZXJmYWNlIFBpcGVEZWNvcmF0b3Ige1xuICAvKipcbiAgICpcbiAgICogRGVjb3JhdG9yIHRoYXQgbWFya3MgYSBjbGFzcyBhcyBwaXBlIGFuZCBzdXBwbGllcyBjb25maWd1cmF0aW9uIG1ldGFkYXRhLlxuICAgKlxuICAgKiBBIHBpcGUgY2xhc3MgbXVzdCBpbXBsZW1lbnQgdGhlIGBQaXBlVHJhbnNmb3JtYCBpbnRlcmZhY2UuXG4gICAqIEZvciBleGFtcGxlLCBpZiB0aGUgbmFtZSBpcyBcIm15UGlwZVwiLCB1c2UgYSB0ZW1wbGF0ZSBiaW5kaW5nIGV4cHJlc3Npb25cbiAgICogc3VjaCBhcyB0aGUgZm9sbG93aW5nOlxuICAgKlxuICAgKiBgYGBcbiAgICoge3sgZXhwIHwgbXlQaXBlIH19XG4gICAqIGBgYFxuICAgKlxuICAgKiBUaGUgcmVzdWx0IG9mIHRoZSBleHByZXNzaW9uIGlzIHBhc3NlZCB0byB0aGUgcGlwZSdzIGB0cmFuc2Zvcm0oKWAgbWV0aG9kLlxuICAgKlxuICAgKiBBIHBpcGUgbXVzdCBiZWxvbmcgdG8gYW4gTmdNb2R1bGUgaW4gb3JkZXIgZm9yIGl0IHRvIGJlIGF2YWlsYWJsZVxuICAgKiB0byBhIHRlbXBsYXRlLiBUbyBtYWtlIGl0IGEgbWVtYmVyIG9mIGFuIE5nTW9kdWxlLFxuICAgKiBsaXN0IGl0IGluIHRoZSBgZGVjbGFyYXRpb25zYCBmaWVsZCBvZiB0aGUgYE5nTW9kdWxlYCBtZXRhZGF0YS5cbiAgICpcbiAgICogQHNlZSBbU3R5bGUgR3VpZGU6IFBpcGUgTmFtZXNdKGd1aWRlL3N0eWxlZ3VpZGUjMDItMDkpXG4gICAqXG4gICAqL1xuICAob2JqOiBQaXBlKTogVHlwZURlY29yYXRvcjtcblxuICAvKipcbiAgICogU2VlIHRoZSBgUGlwZWAgZGVjb3JhdG9yLlxuICAgKi9cbiAgbmV3KG9iajogUGlwZSk6IFBpcGU7XG59XG5cbi8qKlxuICogVHlwZSBvZiB0aGUgUGlwZSBtZXRhZGF0YS5cbiAqXG4gKiBAcHVibGljQXBpXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUGlwZSB7XG4gIC8qKlxuICAgKiBUaGUgcGlwZSBuYW1lIHRvIHVzZSBpbiB0ZW1wbGF0ZSBiaW5kaW5ncy5cbiAgICogVHlwaWNhbGx5IHVzZXMgW2xvd2VyQ2FtZWxDYXNlXShndWlkZS9nbG9zc2FyeSNjYXNlLXR5cGVzKVxuICAgKiBiZWNhdXNlIHRoZSBuYW1lIGNhbm5vdCBjb250YWluIGh5cGhlbnMuXG4gICAqL1xuICBuYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFdoZW4gdHJ1ZSwgdGhlIHBpcGUgaXMgcHVyZSwgbWVhbmluZyB0aGF0IHRoZVxuICAgKiBgdHJhbnNmb3JtKClgIG1ldGhvZCBpcyBpbnZva2VkIG9ubHkgd2hlbiBpdHMgaW5wdXQgYXJndW1lbnRzXG4gICAqIGNoYW5nZS4gUGlwZXMgYXJlIHB1cmUgYnkgZGVmYXVsdC5cbiAgICpcbiAgICogSWYgdGhlIHBpcGUgaGFzIGludGVybmFsIHN0YXRlICh0aGF0IGlzLCB0aGUgcmVzdWx0XG4gICAqIGRlcGVuZHMgb24gc3RhdGUgb3RoZXIgdGhhbiBpdHMgYXJndW1lbnRzKSwgc2V0IGBwdXJlYCB0byBmYWxzZS5cbiAgICogSW4gdGhpcyBjYXNlLCB0aGUgcGlwZSBpcyBpbnZva2VkIG9uIGVhY2ggY2hhbmdlLWRldGVjdGlvbiBjeWNsZSxcbiAgICogZXZlbiBpZiB0aGUgYXJndW1lbnRzIGhhdmUgbm90IGNoYW5nZWQuXG4gICAqL1xuICBwdXJlPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogQW5ndWxhciBwaXBlcyBtYXJrZWQgYXMgYHN0YW5kYWxvbmVgIGRvIG5vdCBuZWVkIHRvIGJlIGRlY2xhcmVkIGluIGFuIE5nTW9kdWxlLiBTdWNoXG4gICAqIHBpcGVzIGRvbid0IGRlcGVuZCBvbiBhbnkgXCJpbnRlcm1lZGlhdGUgY29udGV4dFwiIG9mIGFuIE5nTW9kdWxlIChleC4gY29uZmlndXJlZCBwcm92aWRlcnMpLlxuICAgKlxuICAgKiBNb3JlIGluZm9ybWF0aW9uIGFib3V0IHN0YW5kYWxvbmUgY29tcG9uZW50cywgZGlyZWN0aXZlcyBhbmQgcGlwZXMgY2FuIGJlIGZvdW5kIGluIFt0aGlzXG4gICAqIGd1aWRlXShndWlkZS9zdGFuZGFsb25lLWNvbXBvbmVudHMpLlxuICAgKi9cbiAgc3RhbmRhbG9uZT86IGJvb2xlYW47XG59XG5cbi8qKlxuICogQEFubm90YXRpb25cbiAqIEBwdWJsaWNBcGlcbiAqL1xuZXhwb3J0IGNvbnN0IFBpcGU6IFBpcGVEZWNvcmF0b3IgPSBtYWtlRGVjb3JhdG9yKFxuICAgICdQaXBlJywgKHA6IFBpcGUpID0+ICh7cHVyZTogdHJ1ZSwgLi4ucH0pLCB1bmRlZmluZWQsIHVuZGVmaW5lZCxcbiAgICAodHlwZTogVHlwZTxhbnk+LCBtZXRhOiBQaXBlKSA9PiBjb21waWxlUGlwZSh0eXBlLCBtZXRhKSk7XG5cblxuLyoqXG4gKiBAcHVibGljQXBpXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSW5wdXREZWNvcmF0b3Ige1xuICAvKipcbiAgICogRGVjb3JhdG9yIHRoYXQgbWFya3MgYSBjbGFzcyBmaWVsZCBhcyBhbiBpbnB1dCBwcm9wZXJ0eSBhbmQgc3VwcGxpZXMgY29uZmlndXJhdGlvbiBtZXRhZGF0YS5cbiAgICogVGhlIGlucHV0IHByb3BlcnR5IGlzIGJvdW5kIHRvIGEgRE9NIHByb3BlcnR5IGluIHRoZSB0ZW1wbGF0ZS4gRHVyaW5nIGNoYW5nZSBkZXRlY3Rpb24sXG4gICAqIEFuZ3VsYXIgYXV0b21hdGljYWxseSB1cGRhdGVzIHRoZSBkYXRhIHByb3BlcnR5IHdpdGggdGhlIERPTSBwcm9wZXJ0eSdzIHZhbHVlLlxuICAgKlxuICAgKiBAdXNhZ2VOb3Rlc1xuICAgKlxuICAgKiBZb3UgY2FuIHN1cHBseSBhbiBvcHRpb25hbCBuYW1lIHRvIHVzZSBpbiB0ZW1wbGF0ZXMgd2hlbiB0aGVcbiAgICogY29tcG9uZW50IGlzIGluc3RhbnRpYXRlZCwgdGhhdCBtYXBzIHRvIHRoZVxuICAgKiBuYW1lIG9mIHRoZSBib3VuZCBwcm9wZXJ0eS4gQnkgZGVmYXVsdCwgdGhlIG9yaWdpbmFsXG4gICAqIG5hbWUgb2YgdGhlIGJvdW5kIHByb3BlcnR5IGlzIHVzZWQgZm9yIGlucHV0IGJpbmRpbmcuXG4gICAqXG4gICAqIFRoZSBmb2xsb3dpbmcgZXhhbXBsZSBjcmVhdGVzIGEgY29tcG9uZW50IHdpdGggdHdvIGlucHV0IHByb3BlcnRpZXMsXG4gICAqIG9uZSBvZiB3aGljaCBpcyBnaXZlbiBhIHNwZWNpYWwgYmluZGluZyBuYW1lLlxuICAgKlxuICAgKiBgYGB0eXBlc2NyaXB0XG4gICAqIEBDb21wb25lbnQoe1xuICAgKiAgIHNlbGVjdG9yOiAnYmFuay1hY2NvdW50JyxcbiAgICogICB0ZW1wbGF0ZTogYFxuICAgKiAgICAgQmFuayBOYW1lOiB7e2JhbmtOYW1lfX1cbiAgICogICAgIEFjY291bnQgSWQ6IHt7aWR9fVxuICAgKiAgIGBcbiAgICogfSlcbiAgICogY2xhc3MgQmFua0FjY291bnQge1xuICAgKiAgIC8vIFRoaXMgcHJvcGVydHkgaXMgYm91bmQgdXNpbmcgaXRzIG9yaWdpbmFsIG5hbWUuXG4gICAqICAgQElucHV0KCkgYmFua05hbWU6IHN0cmluZztcbiAgICogICAvLyB0aGlzIHByb3BlcnR5IHZhbHVlIGlzIGJvdW5kIHRvIGEgZGlmZmVyZW50IHByb3BlcnR5IG5hbWVcbiAgICogICAvLyB3aGVuIHRoaXMgY29tcG9uZW50IGlzIGluc3RhbnRpYXRlZCBpbiBhIHRlbXBsYXRlLlxuICAgKiAgIEBJbnB1dCgnYWNjb3VudC1pZCcpIGlkOiBzdHJpbmc7XG4gICAqXG4gICAqICAgLy8gdGhpcyBwcm9wZXJ0eSBpcyBub3QgYm91bmQsIGFuZCBpcyBub3QgYXV0b21hdGljYWxseSB1cGRhdGVkIGJ5IEFuZ3VsYXJcbiAgICogICBub3JtYWxpemVkQmFua05hbWU6IHN0cmluZztcbiAgICogfVxuICAgKlxuICAgKiBAQ29tcG9uZW50KHtcbiAgICogICBzZWxlY3RvcjogJ2FwcCcsXG4gICAqICAgdGVtcGxhdGU6IGBcbiAgICogICAgIDxiYW5rLWFjY291bnQgYmFua05hbWU9XCJSQkNcIiBhY2NvdW50LWlkPVwiNDc0N1wiPjwvYmFuay1hY2NvdW50PlxuICAgKiAgIGBcbiAgICogfSlcbiAgICogY2xhc3MgQXBwIHt9XG4gICAqIGBgYFxuICAgKlxuICAgKiBAc2VlIFtJbnB1dCBhbmQgT3V0cHV0IHByb3BlcnRpZXNdKGd1aWRlL2lucHV0cy1vdXRwdXRzKVxuICAgKi9cbiAgKGJpbmRpbmdQcm9wZXJ0eU5hbWU/OiBzdHJpbmcpOiBhbnk7XG4gIG5ldyhiaW5kaW5nUHJvcGVydHlOYW1lPzogc3RyaW5nKTogYW55O1xufVxuXG4vKipcbiAqIFR5cGUgb2YgbWV0YWRhdGEgZm9yIGFuIGBJbnB1dGAgcHJvcGVydHkuXG4gKlxuICogQHB1YmxpY0FwaVxuICovXG5leHBvcnQgaW50ZXJmYWNlIElucHV0IHtcbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBET00gcHJvcGVydHkgdG8gd2hpY2ggdGhlIGlucHV0IHByb3BlcnR5IGlzIGJvdW5kLlxuICAgKi9cbiAgYmluZGluZ1Byb3BlcnR5TmFtZT86IHN0cmluZztcbn1cblxuLyoqXG4gKiBAQW5ub3RhdGlvblxuICogQHB1YmxpY0FwaVxuICovXG5leHBvcnQgY29uc3QgSW5wdXQ6IElucHV0RGVjb3JhdG9yID1cbiAgICBtYWtlUHJvcERlY29yYXRvcignSW5wdXQnLCAoYmluZGluZ1Byb3BlcnR5TmFtZT86IHN0cmluZykgPT4gKHtiaW5kaW5nUHJvcGVydHlOYW1lfSkpO1xuXG4vKipcbiAqIFR5cGUgb2YgdGhlIE91dHB1dCBkZWNvcmF0b3IgLyBjb25zdHJ1Y3RvciBmdW5jdGlvbi5cbiAqXG4gKiBAcHVibGljQXBpXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgT3V0cHV0RGVjb3JhdG9yIHtcbiAgLyoqXG4gICAqIERlY29yYXRvciB0aGF0IG1hcmtzIGEgY2xhc3MgZmllbGQgYXMgYW4gb3V0cHV0IHByb3BlcnR5IGFuZCBzdXBwbGllcyBjb25maWd1cmF0aW9uIG1ldGFkYXRhLlxuICAgKiBUaGUgRE9NIHByb3BlcnR5IGJvdW5kIHRvIHRoZSBvdXRwdXQgcHJvcGVydHkgaXMgYXV0b21hdGljYWxseSB1cGRhdGVkIGR1cmluZyBjaGFuZ2UgZGV0ZWN0aW9uLlxuICAgKlxuICAgKiBAdXNhZ2VOb3Rlc1xuICAgKlxuICAgKiBZb3UgY2FuIHN1cHBseSBhbiBvcHRpb25hbCBuYW1lIHRvIHVzZSBpbiB0ZW1wbGF0ZXMgd2hlbiB0aGVcbiAgICogY29tcG9uZW50IGlzIGluc3RhbnRpYXRlZCwgdGhhdCBtYXBzIHRvIHRoZVxuICAgKiBuYW1lIG9mIHRoZSBib3VuZCBwcm9wZXJ0eS4gQnkgZGVmYXVsdCwgdGhlIG9yaWdpbmFsXG4gICAqIG5hbWUgb2YgdGhlIGJvdW5kIHByb3BlcnR5IGlzIHVzZWQgZm9yIG91dHB1dCBiaW5kaW5nLlxuICAgKlxuICAgKiBTZWUgYElucHV0YCBkZWNvcmF0b3IgZm9yIGFuIGV4YW1wbGUgb2YgcHJvdmlkaW5nIGEgYmluZGluZyBuYW1lLlxuICAgKlxuICAgKiBAc2VlIFtJbnB1dCBhbmQgT3V0cHV0IHByb3BlcnRpZXNdKGd1aWRlL2lucHV0cy1vdXRwdXRzKVxuICAgKlxuICAgKi9cbiAgKGJpbmRpbmdQcm9wZXJ0eU5hbWU/OiBzdHJpbmcpOiBhbnk7XG4gIG5ldyhiaW5kaW5nUHJvcGVydHlOYW1lPzogc3RyaW5nKTogYW55O1xufVxuXG4vKipcbiAqIFR5cGUgb2YgdGhlIE91dHB1dCBtZXRhZGF0YS5cbiAqXG4gKiBAcHVibGljQXBpXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgT3V0cHV0IHtcbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBET00gcHJvcGVydHkgdG8gd2hpY2ggdGhlIG91dHB1dCBwcm9wZXJ0eSBpcyBib3VuZC5cbiAgICovXG4gIGJpbmRpbmdQcm9wZXJ0eU5hbWU/OiBzdHJpbmc7XG59XG5cbi8qKlxuICogQEFubm90YXRpb25cbiAqIEBwdWJsaWNBcGlcbiAqL1xuZXhwb3J0IGNvbnN0IE91dHB1dDogT3V0cHV0RGVjb3JhdG9yID1cbiAgICBtYWtlUHJvcERlY29yYXRvcignT3V0cHV0JywgKGJpbmRpbmdQcm9wZXJ0eU5hbWU/OiBzdHJpbmcpID0+ICh7YmluZGluZ1Byb3BlcnR5TmFtZX0pKTtcblxuXG5cbi8qKlxuICogVHlwZSBvZiB0aGUgSG9zdEJpbmRpbmcgZGVjb3JhdG9yIC8gY29uc3RydWN0b3IgZnVuY3Rpb24uXG4gKlxuICogQHB1YmxpY0FwaVxuICovXG5leHBvcnQgaW50ZXJmYWNlIEhvc3RCaW5kaW5nRGVjb3JhdG9yIHtcbiAgLyoqXG4gICAqIERlY29yYXRvciB0aGF0IG1hcmtzIGEgRE9NIHByb3BlcnR5IGFzIGEgaG9zdC1iaW5kaW5nIHByb3BlcnR5IGFuZCBzdXBwbGllcyBjb25maWd1cmF0aW9uXG4gICAqIG1ldGFkYXRhLlxuICAgKiBBbmd1bGFyIGF1dG9tYXRpY2FsbHkgY2hlY2tzIGhvc3QgcHJvcGVydHkgYmluZGluZ3MgZHVyaW5nIGNoYW5nZSBkZXRlY3Rpb24sIGFuZFxuICAgKiBpZiBhIGJpbmRpbmcgY2hhbmdlcyBpdCB1cGRhdGVzIHRoZSBob3N0IGVsZW1lbnQgb2YgdGhlIGRpcmVjdGl2ZS5cbiAgICpcbiAgICogQHVzYWdlTm90ZXNcbiAgICpcbiAgICogVGhlIGZvbGxvd2luZyBleGFtcGxlIGNyZWF0ZXMgYSBkaXJlY3RpdmUgdGhhdCBzZXRzIHRoZSBgdmFsaWRgIGFuZCBgaW52YWxpZGBcbiAgICogcHJvcGVydGllcyBvbiB0aGUgRE9NIGVsZW1lbnQgdGhhdCBoYXMgYW4gYG5nTW9kZWxgIGRpcmVjdGl2ZSBvbiBpdC5cbiAgICpcbiAgICogYGBgdHlwZXNjcmlwdFxuICAgKiBARGlyZWN0aXZlKHtzZWxlY3RvcjogJ1tuZ01vZGVsXSd9KVxuICAgKiBjbGFzcyBOZ01vZGVsU3RhdHVzIHtcbiAgICogICBjb25zdHJ1Y3RvcihwdWJsaWMgY29udHJvbDogTmdNb2RlbCkge31cbiAgICogICBASG9zdEJpbmRpbmcoJ2NsYXNzLnZhbGlkJykgZ2V0IHZhbGlkKCkgeyByZXR1cm4gdGhpcy5jb250cm9sLnZhbGlkOyB9XG4gICAqICAgQEhvc3RCaW5kaW5nKCdjbGFzcy5pbnZhbGlkJykgZ2V0IGludmFsaWQoKSB7IHJldHVybiB0aGlzLmNvbnRyb2wuaW52YWxpZDsgfVxuICAgKiB9XG4gICAqXG4gICAqIEBDb21wb25lbnQoe1xuICAgKiAgIHNlbGVjdG9yOiAnYXBwJyxcbiAgICogICB0ZW1wbGF0ZTogYDxpbnB1dCBbKG5nTW9kZWwpXT1cInByb3BcIj5gLFxuICAgKiB9KVxuICAgKiBjbGFzcyBBcHAge1xuICAgKiAgIHByb3A7XG4gICAqIH1cbiAgICogYGBgXG4gICAqXG4gICAqL1xuICAoaG9zdFByb3BlcnR5TmFtZT86IHN0cmluZyk6IGFueTtcbiAgbmV3KGhvc3RQcm9wZXJ0eU5hbWU/OiBzdHJpbmcpOiBhbnk7XG59XG5cbi8qKlxuICogVHlwZSBvZiB0aGUgSG9zdEJpbmRpbmcgbWV0YWRhdGEuXG4gKlxuICogQHB1YmxpY0FwaVxuICovXG5leHBvcnQgaW50ZXJmYWNlIEhvc3RCaW5kaW5nIHtcbiAgLyoqXG4gICAqIFRoZSBET00gcHJvcGVydHkgdGhhdCBpcyBib3VuZCB0byBhIGRhdGEgcHJvcGVydHkuXG4gICAqL1xuICBob3N0UHJvcGVydHlOYW1lPzogc3RyaW5nO1xufVxuXG4vKipcbiAqIEBBbm5vdGF0aW9uXG4gKiBAcHVibGljQXBpXG4gKi9cbmV4cG9ydCBjb25zdCBIb3N0QmluZGluZzogSG9zdEJpbmRpbmdEZWNvcmF0b3IgPVxuICAgIG1ha2VQcm9wRGVjb3JhdG9yKCdIb3N0QmluZGluZycsIChob3N0UHJvcGVydHlOYW1lPzogc3RyaW5nKSA9PiAoe2hvc3RQcm9wZXJ0eU5hbWV9KSk7XG5cblxuLyoqXG4gKiBUeXBlIG9mIHRoZSBIb3N0TGlzdGVuZXIgZGVjb3JhdG9yIC8gY29uc3RydWN0b3IgZnVuY3Rpb24uXG4gKlxuICogQHB1YmxpY0FwaVxuICovXG5leHBvcnQgaW50ZXJmYWNlIEhvc3RMaXN0ZW5lckRlY29yYXRvciB7XG4gIC8qKlxuICAgKiBEZWNvcmF0b3IgdGhhdCBkZWNsYXJlcyBhIERPTSBldmVudCB0byBsaXN0ZW4gZm9yLFxuICAgKiBhbmQgcHJvdmlkZXMgYSBoYW5kbGVyIG1ldGhvZCB0byBydW4gd2hlbiB0aGF0IGV2ZW50IG9jY3Vycy5cbiAgICpcbiAgICogQW5ndWxhciBpbnZva2VzIHRoZSBzdXBwbGllZCBoYW5kbGVyIG1ldGhvZCB3aGVuIHRoZSBob3N0IGVsZW1lbnQgZW1pdHMgdGhlIHNwZWNpZmllZCBldmVudCxcbiAgICogYW5kIHVwZGF0ZXMgdGhlIGJvdW5kIGVsZW1lbnQgd2l0aCB0aGUgcmVzdWx0LlxuICAgKlxuICAgKiBJZiB0aGUgaGFuZGxlciBtZXRob2QgcmV0dXJucyBmYWxzZSwgYXBwbGllcyBgcHJldmVudERlZmF1bHRgIG9uIHRoZSBib3VuZCBlbGVtZW50LlxuICAgKi9cbiAgKGV2ZW50TmFtZTogc3RyaW5nLCBhcmdzPzogc3RyaW5nW10pOiBhbnk7XG4gIG5ldyhldmVudE5hbWU6IHN0cmluZywgYXJncz86IHN0cmluZ1tdKTogYW55O1xufVxuXG4vKipcbiAqIFR5cGUgb2YgdGhlIEhvc3RMaXN0ZW5lciBtZXRhZGF0YS5cbiAqXG4gKiBAcHVibGljQXBpXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSG9zdExpc3RlbmVyIHtcbiAgLyoqXG4gICAqIFRoZSBET00gZXZlbnQgdG8gbGlzdGVuIGZvci5cbiAgICovXG4gIGV2ZW50TmFtZT86IHN0cmluZztcbiAgLyoqXG4gICAqIEEgc2V0IG9mIGFyZ3VtZW50cyB0byBwYXNzIHRvIHRoZSBoYW5kbGVyIG1ldGhvZCB3aGVuIHRoZSBldmVudCBvY2N1cnMuXG4gICAqL1xuICBhcmdzPzogc3RyaW5nW107XG59XG5cbi8qKlxuICogRGVjb3JhdG9yIHRoYXQgYmluZHMgYSBET00gZXZlbnQgdG8gYSBob3N0IGxpc3RlbmVyIGFuZCBzdXBwbGllcyBjb25maWd1cmF0aW9uIG1ldGFkYXRhLlxuICogQW5ndWxhciBpbnZva2VzIHRoZSBzdXBwbGllZCBoYW5kbGVyIG1ldGhvZCB3aGVuIHRoZSBob3N0IGVsZW1lbnQgZW1pdHMgdGhlIHNwZWNpZmllZCBldmVudCxcbiAqIGFuZCB1cGRhdGVzIHRoZSBib3VuZCBlbGVtZW50IHdpdGggdGhlIHJlc3VsdC5cbiAqXG4gKiBJZiB0aGUgaGFuZGxlciBtZXRob2QgcmV0dXJucyBmYWxzZSwgYXBwbGllcyBgcHJldmVudERlZmF1bHRgIG9uIHRoZSBib3VuZCBlbGVtZW50LlxuICpcbiAqIEB1c2FnZU5vdGVzXG4gKlxuICogVGhlIGZvbGxvd2luZyBleGFtcGxlIGRlY2xhcmVzIGEgZGlyZWN0aXZlXG4gKiB0aGF0IGF0dGFjaGVzIGEgY2xpY2sgbGlzdGVuZXIgdG8gYSBidXR0b24gYW5kIGNvdW50cyBjbGlja3MuXG4gKlxuICogYGBgdHNcbiAqIEBEaXJlY3RpdmUoe3NlbGVjdG9yOiAnYnV0dG9uW2NvdW50aW5nXSd9KVxuICogY2xhc3MgQ291bnRDbGlja3Mge1xuICogICBudW1iZXJPZkNsaWNrcyA9IDA7XG4gKlxuICogICBASG9zdExpc3RlbmVyKCdjbGljaycsIFsnJGV2ZW50LnRhcmdldCddKVxuICogICBvbkNsaWNrKGJ0bikge1xuICogICAgIGNvbnNvbGUubG9nKCdidXR0b24nLCBidG4sICdudW1iZXIgb2YgY2xpY2tzOicsIHRoaXMubnVtYmVyT2ZDbGlja3MrKyk7XG4gKiAgIH1cbiAqIH1cbiAqXG4gKiBAQ29tcG9uZW50KHtcbiAqICAgc2VsZWN0b3I6ICdhcHAnLFxuICogICB0ZW1wbGF0ZTogJzxidXR0b24gY291bnRpbmc+SW5jcmVtZW50PC9idXR0b24+JyxcbiAqIH0pXG4gKiBjbGFzcyBBcHAge31cbiAqXG4gKiBgYGBcbiAqXG4gKiBUaGUgZm9sbG93aW5nIGV4YW1wbGUgcmVnaXN0ZXJzIGFub3RoZXIgRE9NIGV2ZW50IGhhbmRsZXIgdGhhdCBsaXN0ZW5zIGZvciBgRW50ZXJgIGtleS1wcmVzc1xuICogZXZlbnRzIG9uIHRoZSBnbG9iYWwgYHdpbmRvd2AuXG4gKiBgYGAgdHNcbiAqIGltcG9ydCB7IEhvc3RMaXN0ZW5lciwgQ29tcG9uZW50IH0gZnJvbSBcIkBhbmd1bGFyL2NvcmVcIjtcbiAqXG4gKiBAQ29tcG9uZW50KHtcbiAqICAgc2VsZWN0b3I6ICdhcHAnLFxuICogICB0ZW1wbGF0ZTogYDxoMT5IZWxsbywgeW91IGhhdmUgcHJlc3NlZCBlbnRlciB7e2NvdW50ZXJ9fSBudW1iZXIgb2YgdGltZXMhPC9oMT4gUHJlc3MgZW50ZXIga2V5XG4gKiB0byBpbmNyZW1lbnQgdGhlIGNvdW50ZXIuXG4gKiAgIDxidXR0b24gKGNsaWNrKT1cInJlc2V0Q291bnRlcigpXCI+UmVzZXQgQ291bnRlcjwvYnV0dG9uPmBcbiAqIH0pXG4gKiBjbGFzcyBBcHBDb21wb25lbnQge1xuICogICBjb3VudGVyID0gMDtcbiAqICAgQEhvc3RMaXN0ZW5lcignd2luZG93OmtleWRvd24uZW50ZXInLCBbJyRldmVudCddKVxuICogICBoYW5kbGVLZXlEb3duKGV2ZW50OiBLZXlib2FyZEV2ZW50KSB7XG4gKiAgICAgdGhpcy5jb3VudGVyKys7XG4gKiAgIH1cbiAqICAgcmVzZXRDb3VudGVyKCkge1xuICogICAgIHRoaXMuY291bnRlciA9IDA7XG4gKiAgIH1cbiAqIH1cbiAqIGBgYFxuICogVGhlIGxpc3Qgb2YgdmFsaWQga2V5IG5hbWVzIGZvciBga2V5ZG93bmAgYW5kIGBrZXl1cGAgZXZlbnRzXG4gKiBjYW4gYmUgZm91bmQgaGVyZTpcbiAqIGh0dHBzOi8vd3d3LnczLm9yZy9UUi9ET00tTGV2ZWwtMy1FdmVudHMta2V5LyNuYW1lZC1rZXktYXR0cmlidXRlLXZhbHVlc1xuICpcbiAqIE5vdGUgdGhhdCBrZXlzIGNhbiBhbHNvIGJlIGNvbWJpbmVkLCBlLmcuIGBASG9zdExpc3RlbmVyKCdrZXlkb3duLnNoaWZ0LmEnKWAuXG4gKlxuICogVGhlIGdsb2JhbCB0YXJnZXQgbmFtZXMgdGhhdCBjYW4gYmUgdXNlZCB0byBwcmVmaXggYW4gZXZlbnQgbmFtZSBhcmVcbiAqIGBkb2N1bWVudDpgLCBgd2luZG93OmAgYW5kIGBib2R5OmAuXG4gKlxuICogQEFubm90YXRpb25cbiAqIEBwdWJsaWNBcGlcbiAqL1xuZXhwb3J0IGNvbnN0IEhvc3RMaXN0ZW5lcjogSG9zdExpc3RlbmVyRGVjb3JhdG9yID1cbiAgICBtYWtlUHJvcERlY29yYXRvcignSG9zdExpc3RlbmVyJywgKGV2ZW50TmFtZT86IHN0cmluZywgYXJncz86IHN0cmluZ1tdKSA9PiAoe2V2ZW50TmFtZSwgYXJnc30pKTtcbiJdfQ==