/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import '../util/ng_dev_mode';
import { RuntimeError } from '../errors';
import { stringify } from '../util/stringify';
import { resolveForwardRef } from './forward_ref';
import { getInjectImplementation, injectRootLimpMode } from './inject_switch';
import { InjectFlags } from './interface/injector';
const _THROW_IF_NOT_FOUND = {};
export const THROW_IF_NOT_FOUND = _THROW_IF_NOT_FOUND;
/*
 * Name of a property (that we patch onto DI decorator), which is used as an annotation of which
 * InjectFlag this decorator represents. This allows to avoid direct references to the DI decorators
 * in the code, thus making them tree-shakable.
 */
const DI_DECORATOR_FLAG = '__NG_DI_FLAG__';
export const NG_TEMP_TOKEN_PATH = 'ngTempTokenPath';
const NG_TOKEN_PATH = 'ngTokenPath';
const NEW_LINE = /\n/gm;
const NO_NEW_LINE = 'ɵ';
export const SOURCE = '__source';
/**
 * Current injector value used by `inject`.
 * - `undefined`: it is an error to call `inject`
 * - `null`: `inject` can be called but there is no injector (limp-mode).
 * - Injector instance: Use the injector for resolution.
 */
let _currentInjector = undefined;
export function setCurrentInjector(injector) {
    const former = _currentInjector;
    _currentInjector = injector;
    return former;
}
export function injectInjectorOnly(token, flags = InjectFlags.Default) {
    if (_currentInjector === undefined) {
        throw new RuntimeError(-203 /* RuntimeErrorCode.MISSING_INJECTION_CONTEXT */, ngDevMode &&
            `inject() must be called from an injection context (a constructor, a factory function or a field initializer)`);
    }
    else if (_currentInjector === null) {
        return injectRootLimpMode(token, undefined, flags);
    }
    else {
        return _currentInjector.get(token, flags & InjectFlags.Optional ? null : undefined, flags);
    }
}
export function ɵɵinject(token, flags = InjectFlags.Default) {
    return (getInjectImplementation() || injectInjectorOnly)(resolveForwardRef(token), flags);
}
/**
 * Throws an error indicating that a factory function could not be generated by the compiler for a
 * particular class.
 *
 * The name of the class is not mentioned here, but will be in the generated factory function name
 * and thus in the stack trace.
 *
 * @codeGenApi
 */
export function ɵɵinvalidFactoryDep(index) {
    throw new RuntimeError(202 /* RuntimeErrorCode.INVALID_FACTORY_DEPENDENCY */, ngDevMode &&
        `This constructor is not compatible with Angular Dependency Injection because its dependency at index ${index} of the parameter list is invalid.
This can happen if the dependency type is a primitive like a string or if an ancestor of this class is missing an Angular decorator.

Please check that 1) the type for the parameter at index ${index} is correct and 2) the correct Angular decorators are defined for this class and its ancestors.`);
}
/**
 * Injects a token from the currently active injector.
 * `inject` is only supported during instantiation of a dependency by the DI system. It can be used
 * during:
 * - Construction (via the `constructor`) of a class being instantiated by the DI system, such
 * as an `@Injectable` or `@Component`.
 * - In the initializer for fields of such classes.
 * - In the factory function specified for `useFactory` of a `Provider` or an `@Injectable`.
 * - In the `factory` function specified for an `InjectionToken`.
 *
 * @param token A token that represents a dependency that should be injected.
 * @param flags Optional flags that control how injection is executed.
 * The flags correspond to injection strategies that can be specified with
 * parameter decorators `@Host`, `@Self`, `@SkipSef`, and `@Optional`.
 * @returns the injected value if operation is successful, `null` otherwise.
 * @throws if called outside of a supported context.
 *
 * @usageNotes
 * In practice the `inject()` calls are allowed in a constructor, a constructor parameter and a
 * field initializer:
 *
 * ```typescript
 * @Injectable({providedIn: 'root'})
 * export class Car {
 *   radio: Radio|undefined;
 *   // OK: field initializer
 *   spareTyre = inject(Tyre);
 *
 *   constructor() {
 *     // OK: constructor body
 *     this.radio = inject(Radio);
 *   }
 * }
 * ```
 *
 * It is also legal to call `inject` from a provider's factory:
 *
 * ```typescript
 * providers: [
 *   {provide: Car, useFactory: () => {
 *     // OK: a class factory
 *     const engine = inject(Engine);
 *     return new Car(engine);
 *   }}
 * ]
 * ```
 *
 * Calls to the `inject()` function outside of the class creation context will result in error. Most
 * notably, calls to `inject()` are disallowed after a class instance was created, in methods
 * (including lifecycle hooks):
 *
 * ```typescript
 * @Component({ ... })
 * export class CarComponent {
 *   ngOnInit() {
 *     // ERROR: too late, the component instance was already created
 *     const engine = inject(Engine);
 *     engine.start();
 *   }
 * }
 * ```
 *
 * @publicApi
 */
export function inject(token, flags = InjectFlags.Default) {
    return ɵɵinject(token, flags);
}
export function injectArgs(types) {
    const args = [];
    for (let i = 0; i < types.length; i++) {
        const arg = resolveForwardRef(types[i]);
        if (Array.isArray(arg)) {
            if (arg.length === 0) {
                throw new RuntimeError(900 /* RuntimeErrorCode.INVALID_DIFFER_INPUT */, ngDevMode && 'Arguments array must have arguments.');
            }
            let type = undefined;
            let flags = InjectFlags.Default;
            for (let j = 0; j < arg.length; j++) {
                const meta = arg[j];
                const flag = getInjectFlag(meta);
                if (typeof flag === 'number') {
                    // Special case when we handle @Inject decorator.
                    if (flag === -1 /* DecoratorFlags.Inject */) {
                        type = meta.token;
                    }
                    else {
                        flags |= flag;
                    }
                }
                else {
                    type = meta;
                }
            }
            args.push(ɵɵinject(type, flags));
        }
        else {
            args.push(ɵɵinject(arg));
        }
    }
    return args;
}
/**
 * Attaches a given InjectFlag to a given decorator using monkey-patching.
 * Since DI decorators can be used in providers `deps` array (when provider is configured using
 * `useFactory`) without initialization (e.g. `Host`) and as an instance (e.g. `new Host()`), we
 * attach the flag to make it available both as a static property and as a field on decorator
 * instance.
 *
 * @param decorator Provided DI decorator.
 * @param flag InjectFlag that should be applied.
 */
export function attachInjectFlag(decorator, flag) {
    decorator[DI_DECORATOR_FLAG] = flag;
    decorator.prototype[DI_DECORATOR_FLAG] = flag;
    return decorator;
}
/**
 * Reads monkey-patched property that contains InjectFlag attached to a decorator.
 *
 * @param token Token that may contain monkey-patched DI flags property.
 */
export function getInjectFlag(token) {
    return token[DI_DECORATOR_FLAG];
}
export function catchInjectorError(e, token, injectorErrorName, source) {
    const tokenPath = e[NG_TEMP_TOKEN_PATH];
    if (token[SOURCE]) {
        tokenPath.unshift(token[SOURCE]);
    }
    e.message = formatError('\n' + e.message, tokenPath, injectorErrorName, source);
    e[NG_TOKEN_PATH] = tokenPath;
    e[NG_TEMP_TOKEN_PATH] = null;
    throw e;
}
export function formatError(text, obj, injectorErrorName, source = null) {
    text = text && text.charAt(0) === '\n' && text.charAt(1) == NO_NEW_LINE ? text.slice(2) : text;
    let context = stringify(obj);
    if (Array.isArray(obj)) {
        context = obj.map(stringify).join(' -> ');
    }
    else if (typeof obj === 'object') {
        let parts = [];
        for (let key in obj) {
            if (obj.hasOwnProperty(key)) {
                let value = obj[key];
                parts.push(key + ':' + (typeof value === 'string' ? JSON.stringify(value) : stringify(value)));
            }
        }
        context = `{${parts.join(', ')}}`;
    }
    return `${injectorErrorName}${source ? '(' + source + ')' : ''}[${context}]: ${text.replace(NEW_LINE, '\n  ')}`;
}
//# sourceMappingURL=data:application/json;base64,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