/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
import { assertLContainer } from '../render3/assert';
import { createLView, renderView } from '../render3/instructions/shared';
import { DECLARATION_LCONTAINER, QUERIES } from '../render3/interfaces/view';
import { getCurrentTNode, getLView } from '../render3/state';
import { ViewRef as R3_ViewRef } from '../render3/view_ref';
import { assertDefined } from '../util/assert';
import { createElementRef } from './element_ref';
/**
 * Represents an embedded template that can be used to instantiate embedded views.
 * To instantiate embedded views based on a template, use the `ViewContainerRef`
 * method `createEmbeddedView()`.
 *
 * Access a `TemplateRef` instance by placing a directive on an `<ng-template>`
 * element (or directive prefixed with `*`). The `TemplateRef` for the embedded view
 * is injected into the constructor of the directive,
 * using the `TemplateRef` token.
 *
 * You can also use a `Query` to find a `TemplateRef` associated with
 * a component or a directive.
 *
 * @see `ViewContainerRef`
 * @see [Navigate the Component Tree with DI](guide/dependency-injection-navtree)
 *
 * @publicApi
 */
export class TemplateRef {
}
/**
 * @internal
 * @nocollapse
 */
TemplateRef.__NG_ELEMENT_ID__ = injectTemplateRef;
const ViewEngineTemplateRef = TemplateRef;
// TODO(alxhub): combine interface and implementation. Currently this is challenging since something
// in g3 depends on them being separate.
const R3TemplateRef = class TemplateRef extends ViewEngineTemplateRef {
    constructor(_declarationLView, _declarationTContainer, elementRef) {
        super();
        this._declarationLView = _declarationLView;
        this._declarationTContainer = _declarationTContainer;
        this.elementRef = elementRef;
    }
    createEmbeddedView(context, injector) {
        const embeddedTView = this._declarationTContainer.tViews;
        const embeddedLView = createLView(this._declarationLView, embeddedTView, context, 16 /* LViewFlags.CheckAlways */, null, embeddedTView.declTNode, null, null, null, null, injector || null);
        const declarationLContainer = this._declarationLView[this._declarationTContainer.index];
        ngDevMode && assertLContainer(declarationLContainer);
        embeddedLView[DECLARATION_LCONTAINER] = declarationLContainer;
        const declarationViewLQueries = this._declarationLView[QUERIES];
        if (declarationViewLQueries !== null) {
            embeddedLView[QUERIES] = declarationViewLQueries.createEmbeddedView(embeddedTView);
        }
        renderView(embeddedTView, embeddedLView, context);
        return new R3_ViewRef(embeddedLView);
    }
};
/**
 * Creates a TemplateRef given a node.
 *
 * @returns The TemplateRef instance to use
 */
export function injectTemplateRef() {
    return createTemplateRef(getCurrentTNode(), getLView());
}
/**
 * Creates a TemplateRef and stores it on the injector.
 *
 * @param hostTNode The node on which a TemplateRef is requested
 * @param hostLView The `LView` to which the node belongs
 * @returns The TemplateRef instance or null if we can't create a TemplateRef on a given node type
 */
export function createTemplateRef(hostTNode, hostLView) {
    if (hostTNode.type & 4 /* TNodeType.Container */) {
        ngDevMode && assertDefined(hostTNode.tViews, 'TView must be allocated');
        return new R3TemplateRef(hostLView, hostTNode, createElementRef(hostTNode, hostLView));
    }
    return null;
}
//# sourceMappingURL=data:application/json;base64,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